package org.deft.language.java.parser;

import java.io.BufferedReader;
import java.io.CharArrayReader;
import java.io.File;
import java.io.FileReader;
import java.io.IOException;
import java.nio.CharBuffer;
import java.util.HashMap;
import java.util.Map;

import org.antlr.runtime.ANTLRReaderStream;
import org.antlr.runtime.CommonTokenStream;
import org.antlr.runtime.ParserRuleReturnScope;
import org.antlr.runtime.RecognitionException;
import org.apache.log4j.Logger;
import org.deft.language.java.parser.antlradapter.DeftTreeAdaptor;
import org.deft.repository.ast.AstCombiner;
import org.deft.repository.ast.MergeConfiguration;
import org.deft.repository.ast.TreeNode;
import org.deft.repository.ast.TreeNodeRoot;
import org.eclipse.jdt.core.compiler.IProblem;
import org.eclipse.jdt.core.dom.AST;
import org.eclipse.jdt.core.dom.ASTParser;
import org.eclipse.jdt.core.dom.CompilationUnit;

public class JavaParser {

	private static final String defaultSourceLevel = "1.5";

	private static Logger logger = Logger.getLogger(JavaParser.class);

	private String sourceLevel = defaultSourceLevel;

	private CharBuffer getCharBuffer(BufferedReader reader) throws IOException {
		CharBuffer cb = CharBuffer.allocate(1024 * 8);
		char[] buf = new char[1024];
		int length;
		int offset = 0;
		while ((length = reader.read(buf)) != -1) {
			cb.put(buf, 0, length);
			offset += length;
			if (offset == cb.capacity()) {
				CharBuffer tmp = CharBuffer.allocate(cb.capacity() * 2);
				cb.position(0);
				tmp.put(cb);
				cb = tmp;
			}
		}
		return CharBuffer.allocate(offset).put(cb.array(), 0, offset);
	}

	public TreeNodeRoot getAst(CharBuffer content) throws DeftParseException {
		// make position converter
		PositionConverter pConv = getPositionConverter(content);
		TreeNode javaTree = getJavaTree(content, pConv);
		TreeNode commentTree = getCommentTree(content, pConv);

		// merge Java and comment tree
		MergeConfiguration config = new MergeConfiguration();
		AstCombiner combiner = new AstCombiner();
		combiner.combineTrees(javaTree, commentTree, config);

		// TreeNodeRoot root = visitor.getRoot();
		TreeNodeRoot root = new TreeNodeRoot(javaTree);
		return root;
	}

	private PositionConverter getPositionConverter(CharBuffer content) {
		PositionConverter pConv = null;
		BufferedReader reader = new BufferedReader(new CharArrayReader(content
				.array()));
		try {
			pConv = new PositionConverter(reader);
		} catch (IOException e) {
			throw new RuntimeException(
					"Error while trying to read input stream.", e);
		}
		return pConv;
	}

	private TreeNode getJavaTree(CharBuffer content, PositionConverter pConv)
			throws DeftParseException {
		ASTParser parser = ASTParser.newParser(AST.JLS3);
		parser.setKind(ASTParser.K_COMPILATION_UNIT);
		parser.setSource(content.array());

		// set options AFTER source, otherwise it is discarded
		Map<String, String> options = new HashMap<String, String>();
		// compiler source level is per default 1.3, change it
		options.put("org.eclipse.jdt.core.compiler.source", sourceLevel);
		parser.setCompilerOptions(options);

		CompilationUnit node = (CompilationUnit) parser.createAST(null);
		AstConverterVisitor visitor = new AstConverterVisitor(content.array(),
				pConv);
		node.accept(visitor);
		// if there were parse errors
		if (node.getProblems().length > 0) {
			IProblem firstProblem = node.getProblems()[0];
			String errorMsg = "Line [" + firstProblem.getSourceLineNumber()
					+ "]: " + firstProblem.getMessage();
			DeftParseException dpe = new DeftParseException(errorMsg);
			throw dpe;
		}
		TreeNode tree = visitor.getTree();
		return tree;
	}

	private TreeNode getCommentTree(CharBuffer content, PositionConverter pConv) {
		TreeNode commentTree = null;
		try {
			CStyleCommentsLexer commentLexer = new CStyleCommentsLexer(
					new ANTLRReaderStream(new CharArrayReader(content.array())));
			CommonTokenStream commentTokens = new CommonTokenStream(
					commentLexer);
			CStyleCommentsParser commentParser = new CStyleCommentsParser(
					commentTokens);
			commentParser.setTreeAdaptor(new DeftTreeAdaptor(
					new CStyleCommentsMapping(pConv)));
			ParserRuleReturnScope commentRet = commentParser.compilationunit();
			commentTree = (TreeNode) commentRet.getTree();
		} catch (IOException e) {

			e.printStackTrace();
		} catch (RecognitionException e) {

			e.printStackTrace();
		}
		return commentTree;
	}

	public TreeNodeRoot getAst(File file) throws DeftParseException {
		CharBuffer content = getContent(file);
		return getAst(content);
	}

	public CharBuffer getContent(File file) {
		try {
			BufferedReader reader = new BufferedReader(new FileReader(file));
			return getCharBuffer(reader);
		} catch (IOException e) {
			logger.error(e.getMessage(), e);
		}

		return null;
	}

	public String getSourceLevel() {
		return sourceLevel;
	}

	/**
	 * Sets the source level to use. The default source level is "1.5".
	 * 
	 * @param sourceLevel
	 *            The source level to use.
	 */
	public void setSourceLevel(String sourceLevel) {
		this.sourceLevel = sourceLevel;
	}
}
