package org.deft.extension.test;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.fail;

import java.util.LinkedList;
import java.util.List;

import org.deft.extension.test.testutil.TestLoader;
import org.deft.extension.test.testutil.TestUtil;
import org.deft.extension.tools.astlayouter.ASTLayouter;
import org.deft.extension.tools.astlayouter.InvalidArgumentException;
import org.deft.repository.ast.Token;
import org.deft.repository.ast.TokenNode;
import org.deft.repository.ast.TreeNode;
import org.junit.Test;

public class ASTLayouterTest {

	private String lineBreak = "\n";
	private String testRoot = "resources/test/ASTLayouterTest/";

	// Init function is shorter with this private parameter
	private String file;
	private Integer lineOfToken;
	private Integer colOfToken;
	private Integer offset;
	private String expected;
	private Integer startLine;
	private Integer endLine;
	private TreeNode ast;
	private List<TokenNode> nodes;
	private Token testToken;
	private Token startToken;
	private Token endToken;
	private ASTLayouter layouter;
	private String xpath;

	@Test
	public void testOffsetRepair() throws Exception {

		String extDir = "g:/beleg/org.deft.extension/src/org/deft/extension/";
		String[] files = { "resources/SimpleTestFile.java",
				"resources/CommentTestFile.java",
				extDir + "tools/astlayouter/ASTLayouter.java",
				extDir + "decoration/style/styleDecorator.java",
				"resources/JavaParserUNIX.java", "resources/JavaParserWIN.java" };

		for (String file : files) {
			TreeNode ast = org.deft.extension.test.testutil.TestUtil
					.readFileInAST(file);
			List<TokenNode> nodes = ast.serialize();

			String expected = TestUtil.tokenNodeListToString(nodes, lineBreak);

			ASTLayouter layouter = new ASTLayouter(nodes);
			layouter.repairOffset();

			String actual = TestUtil.tokenNodeListToString(nodes, lineBreak);

			assertEquals(file + " failed", expected, actual);
		}
	}

	@Test
	public void testInsertSpacesBefore() throws Exception {

		TestLoader tests = new TestLoader(testRoot
				+ "insertSpaceBefore.testcase");
		for (org.deft.extension.test.testutil.Test test : tests) {
			initTestWithOffset(test);
			layouter.insertSpacesBefore(testToken, offset);
			assertTest(test);
		}

	}

	@Test
	public void testInsertSpacesAfter() throws Exception {

		TestLoader tests = new TestLoader(testRoot
				+ "insertSpaceAfter.testcase");
		for (org.deft.extension.test.testutil.Test test : tests) {
			initTestWithOffset(test);
			layouter.insertSpacesAfter(testToken, offset);
			assertTest(test);
		}

		try {
			layouter.insertSpacesAfter(testToken, -1);
			fail("Exception expected");
		} catch (InvalidArgumentException e) {

		}
	}

	@Test
	public void testInsertLinesBefore() throws Exception {
		TestLoader tests = new TestLoader(testRoot
				+ "insertLinesBefore.testcase");
		for (org.deft.extension.test.testutil.Test test : tests) {
			initTestWithOffset(test);
			layouter.insertLinesBefore(testToken, offset);
			assertTest(test);
		}
		try {
			layouter.insertLinesBefore(testToken, -1);
			fail("Exception expected");
		} catch (InvalidArgumentException e) {

		}

	}

	@Test
	public void testInsertLinesAfter() throws Exception {
		TestLoader tests = new TestLoader(testRoot
				+ "insertLinesAfter.testcase");
		for (org.deft.extension.test.testutil.Test test : tests) {
			initTestWithOffset(test);
			layouter.insertLinesAfter(testToken, offset);
			assertTest(test);
		}
		try {
			layouter.insertLinesAfter(testToken, -1);
			fail("Exception expected");
		} catch (InvalidArgumentException e) {

		}
	}

	@Test
	public void testRemoveLineBreakAfter() throws Exception {
		TestLoader tests = new TestLoader(testRoot
				+ "removeLineBreakAfter.testcase");
		for (org.deft.extension.test.testutil.Test test : tests) {
			initTestWithoutOffset(test);
			layouter.removeLineBreakAfter(testToken);
			assertTest(test);
		}
	}

	@Test
	public void testRemoveLineBreakBefore() throws Exception {
		TestLoader tests = new TestLoader(testRoot
				+ "removeLineBreakBefore.testcase");
		for (org.deft.extension.test.testutil.Test test : tests) {
			initTestWithoutOffset(test);
			layouter.removeLineBreakBefore(testToken);
			assertTest(test);
		}
	}

	@Test
	public void testTrim() throws Exception {
		TestLoader tests = new TestLoader(testRoot + "trim.testcase");
		for (org.deft.extension.test.testutil.Test test : tests) {
			initTestWithoutOffset(test);
			layouter.trim(testToken);
			assertTest(test);
		}
	}

	@Test
	public void testTrimLeft() throws Exception {
		TestLoader tests = new TestLoader(testRoot + "trimLeft.testcase");
		for (org.deft.extension.test.testutil.Test test : tests) {
			initTestWithoutOffset(test);
			layouter.trimLeft(testToken);
			assertTest(test);
		}
	}

	@Test
	public void testTrimRight() throws Exception {
		TestLoader tests = new TestLoader(testRoot + "trimRight.testcase");
		for (org.deft.extension.test.testutil.Test test : tests) {
			initTestWithoutOffset(test);
			layouter.trimRight(testToken);
			assertTest(test);
		}
	}

	@Test
	public void testTrimLinesBefore() throws Exception {
		TestLoader tests = new TestLoader(testRoot + "trimLinesBefore.testcase");
		for (org.deft.extension.test.testutil.Test test : tests) {
			initTestWithoutOffset(test);
			layouter.trimLinesBefore(testToken);
			assertTest(test);
		}
	}

	@Test
	public void testTrimLinesAfter() throws Exception {
		TestLoader tests = new TestLoader(testRoot + "trimLinesAfter.testcase");
		for (org.deft.extension.test.testutil.Test test : tests) {
			initTestWithoutOffset(test);
			layouter.trimLinesAfter(testToken);
			assertTest(test);
		}
	}

	@Test
	public void testInsertLineBreakBefore() throws Exception {
		TestLoader tests = new TestLoader(testRoot
				+ "insertLineBreakBefore.testcase");
		for (org.deft.extension.test.testutil.Test test : tests) {
			initTestWithoutOffset(test);
			layouter.insertLineBreakBefore(testToken);
			assertTest(test);
		}
	}

	@Test
	public void testInsertLineBreakAfter() throws Exception {
		TestLoader tests = new TestLoader(testRoot
				+ "insertLineBreakAfter.testcase");
		for (org.deft.extension.test.testutil.Test test : tests) {
			initTestWithoutOffset(test);
			layouter.insertLineBreakAfter(testToken);
			assertTest(test);
		}
	}

	@Test
	public void testIndent() throws Exception {
		TestLoader tests = new TestLoader(testRoot + "indent.testcase");
		for (org.deft.extension.test.testutil.Test test : tests) {
			initTestWithRange(test);
			layouter.indentRelative(startToken, endToken, offset);
			assertTest(test);
		}
	}

	@Test
	public void testSimpleCache() throws Exception {
		file = "resources/SimpleTestFile.java";
		ast = org.deft.extension.test.testutil.TestUtil.readFileInAST(file);
		nodes = ast.serialize();
		layouter = new ASTLayouter(nodes);
		layouter.insertLinesBefore(4);
		layouter.repairOffset();

		nodes = ast.serialize();
		layouter = new ASTLayouter(nodes);
		layouter.insertLinesBefore(4);
		layouter.repairOffset();

		String actual = TestUtil.tokenNodeListToString(nodes, lineBreak);
		expected = "PACKAGE:9:1-9:8-7 16\nIdentifier:9:9-9:12-3 24\nDOT:9:12-9:13-1 27\nIdentifier:9:13-9:17-4 28\nDOT:9:17-9:18-1 32\nIdentifier:9:18-9:23-5 33\nDOT:9:23-9:24-1 38\nIdentifier:9:24-9:31-7 39\nSEMICOLON:9:31-9:32-1 46\nPUBLIC:11:1-11:7-6 51\nCLASS:11:8-11:13-5 58\nIdentifier:11:14-11:20-6 64\nLBRACE:11:21-11:22-1 71\nComment:13:2-15:4-26 77\nPUBLIC:16:2-16:8-6 106\nSTATIC:16:9-16:15-6 113\nVOID:16:16-16:20-4 120\nIdentifier:16:21-16:25-4 125\nLPAREN:16:25-16:26-1 129\nIdentifier:16:26-16:32-6 130\nLBRACKET:16:32-16:33-1 136\nRBRACKET:16:33-16:34-1 137\nIdentifier:16:35-16:39-4 139\nRPAREN:16:39-16:40-1 143\nLBRACE:16:41-16:42-1 145\nComment:17:3-17:37-34 150\nRBRACE:19:2-19:3-1 189\nRBRACE:21:1-21:2-1 194";
		assertEquals("SimpleCache failed", expected, actual);
	}

	private Token getFirstTokenAfterPosition(List<TokenNode> nodes, int line,
			int col) {
		for (TokenNode node : nodes) {
			Token t = node.getToken();
			if (t.getLine() >= line && t.getCol() >= col) {
				return t;
			}
		}
		return null;
	}

	private void initTestWithOffset(org.deft.extension.test.testutil.Test test)
			throws Exception {
		initCore(test);

		lineOfToken = test.getParamterAsInteger("lineOfToken");
		colOfToken = test.getParamterAsInteger("colOfToken");
		offset = test.getParamterAsInteger("offset");
		testToken = getFirstTokenAfterPosition(nodes, lineOfToken, colOfToken);
	}

	private void initTestWithoutOffset(
			org.deft.extension.test.testutil.Test test) throws Exception {
		initCore(test);

		lineOfToken = test.getParamterAsInteger("lineOfToken");
		colOfToken = test.getParamterAsInteger("colOfToken");
		testToken = getFirstTokenAfterPosition(nodes, lineOfToken, colOfToken);
	}

	private void initTestWithRange(org.deft.extension.test.testutil.Test test)
			throws Exception {
		initCore(test);

		startLine = test.getParamterAsInteger("startLine");
		endLine = test.getParamterAsInteger("endLine");
		startToken = getFirstTokenAfterPosition(nodes, startLine, 1);
		endToken = getFirstTokenAfterPosition(nodes, endLine, 1);
		offset = test.getParamterAsInteger("offset");
	}

	private void assertTest(org.deft.extension.test.testutil.Test test) {
		layouter.repairOffset();
		String actual = TestUtil.tokenNodeListToString(nodes, lineBreak);
		assertEquals(test.getName() + " failed", expected, actual);
	}

	private void initCore(org.deft.extension.test.testutil.Test test)
			throws Exception {
		file = test.getParamterAsString("input");
		expected = test.getParamterAsString("expected").trim();

		ast = org.deft.extension.test.testutil.TestUtil.readFileInAST(file);
		nodes = ast.serialize();
		layouter = new ASTLayouter(nodes);
	}

	@Test
	public void testReset() throws Exception {

		TestLoader tests = new TestLoader(testRoot + "reset.testcase");
		for (org.deft.extension.test.testutil.Test test : tests) {

			file = test.getParamterAsString("input");
			expected = test.getParamterAsString("expected").trim();
			xpath = test.getParamterAsString("xpath");

			ast = org.deft.extension.test.testutil.TestUtil.readFileInAST(file);

			List<TreeNode> trees = ast.executeXPathQuery(xpath);
			List<TokenNode> nodes = new LinkedList<TokenNode>();
			if (trees.size() != 1) {
				fail("Zuviele Trees");
			}
			for (TreeNode t : trees) {
				t.serialize(nodes);
			}

			layouter = new ASTLayouter(nodes);

			layouter.reset();
			layouter.repairOffset();

			String actual = TestUtil.tokenNodeListToString(nodes, lineBreak);
			assertEquals(test.getName() + " failed", expected, actual);
		}

	}

}
