package org.deft.extension.test;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertTrue;

import java.io.File;
import java.io.FileReader;
import java.util.LinkedHashMap;

import org.custommonkey.xmlunit.Diff;
import org.custommonkey.xmlunit.XMLUnit;
import org.deft.extension.decoration.protect.ProtectedDecorator;
import org.deft.extension.decoration.protect.ProtectedDecoratorInvoker;
import org.deft.extension.decoration.style.StyleDecorator;
import org.deft.extension.decoration.style.StyleDecoratorInvoker;
import org.deft.extension.persistence.Decoration;
import org.deft.extension.persistence.Modification;
import org.deft.extension.persistence.Persistence;
import org.deft.extension.test.testutil.TestUtil;
import org.deft.extension.tools.astlayouter.ASTLayouterInvoker;
import org.deft.extension.tools.astmodifier.ASTModifierInvoker;
import org.deft.extension.tools.astmodifier.ModifierException;
import org.deft.repository.ast.TreeNode;
import org.deft.repository.ast.decoration.DecoratorSelection;
import org.junit.Test;

public class PersistenceTest {

	@Test
	public void testLoadStore() throws Exception {
		File in = new File("resources/sample.xml");
		File out = new File("resources/sample2.xml");
		Persistence p = new Persistence(in);
		p.store(out);

		Diff diff = new Diff(new FileReader(in), new FileReader(out));
		XMLUnit.setIgnoreWhitespace(true);
		assertTrue(diff.toString(), diff.identical());
	}

	@Test
	public void testApply() throws Exception {

		TreeNode ast = TestUtil.readFileInAST("resources/SimpleTestFile.java");
		Persistence p = new Persistence(new File("resources/sample.xml"));
		DecoratorSelection ds = new DecoratorSelection();
		ds.addDecorator(new ProtectedDecorator());
		ds.addDecorator(new StyleDecorator());

		p.apply(ast);

		String expected = TestUtil.readFileInString("resources/result.xml");
		String actual = TestUtil.astToXMLString(ast, ds);
		assertEquals(expected, actual);

	}

	@Test(expected = ModifierException.class)
	public void testInsertFail() throws Exception {

		Persistence p = new Persistence();
		LinkedHashMap<String, String> params = new LinkedHashMap<String, String>();
		params.put("method", "appendAfter");
		params.put("location", "/CompilationUnit/PackageDeclaration/SEMICOLON");
		params.put("insert", "resources/SimpleTestFile.java||/");
		Modification m = new Modification(new ASTModifierInvoker(), params);
		p.add(m);

		TreeNode ast = TestUtil.readFileInAST("resources/SimpleTestFile.java");
		p.apply(ast);

	}

	@Test
	public void testManuelBuildAndApply() throws Exception {

		Persistence p = new Persistence();
		// Same as sample.xml
		LinkedHashMap<String, String> params1 = new LinkedHashMap<String, String>();
		ASTLayouterInvoker inv1 = new ASTLayouterInvoker();
		params1.put("method", "insertSpacesBefore");
		params1.put("location", "/CompilationUnit/PackageDeclaration/PACKAGE");
		params1.put("offset", "3");
		Modification m1 = new Modification(inv1, params1);
		p.add(m1);

		LinkedHashMap<String, String> params2 = new LinkedHashMap<String, String>();
		params2.put("method", "trim");
		params2.put("location", "/CompilationUnit/ClassDeclaration/Modifiers");
		Modification m2 = new Modification(inv1, params2);
		p.add(m2);

		LinkedHashMap<String, String> params3 = new LinkedHashMap<String, String>();
		ASTModifierInvoker inv2 = new ASTModifierInvoker();
		params3.put("method", "appendAfter");
		params3.put("location", "/CompilationUnit/PackageDeclaration/PACKAGE");
		params3.put("spacesAfter", "2");
		params3.put("insert", "resources/SimpleTestFile.java||//PACKAGE");
		Modification m3 = new Modification(inv2, params3);
		p.add(m3);

		LinkedHashMap<String, String> params4 = new LinkedHashMap<String, String>();
		params4.put("method", "appendAfter");
		params4
				.put("location",
						"/CompilationUnit/PackageDeclaration/SEMICOLON");
		params4.put("insert", "resources/SimpleTestFile.java||//PACKAGE");
		Modification m4 = new Modification(inv2, params4);
		p.add(m4);

		LinkedHashMap<String, String> params5 = new LinkedHashMap<String, String>();
		StyleDecoratorInvoker inv3 = new StyleDecoratorInvoker();
		params5.put("location", "//Identifier");
		params5
				.put("container",
						"org.deft.extension.decoration.style.container.CssDeclarationStyleContainer");
		params5.put("value", "font-weight:bold;");
		Decoration d1 = new Decoration(inv3, params5);
		p.add(d1);

		LinkedHashMap<String, String> params6 = new LinkedHashMap<String, String>();
		ProtectedDecoratorInvoker inv4 = new ProtectedDecoratorInvoker();
		params6.put("location", "//MethodDeclaration");
		Decoration d2 = new Decoration(inv4, params6);
		p.add(d2);

		LinkedHashMap<String, String> params7 = new LinkedHashMap<String, String>();
		params7.put("method", "changeToComment");
		params7
				.put("location",
						"/CompilationUnit/ClassDeclaration/Body/RBRACE");
		params7
				.put("commentType",
						"org.deft.extension.tools.astmodifier.comment.JavaBlockCommentVisitor");
		Modification m5 = new Modification(inv2, params7);
		p.add(m5);

		TreeNode ast = TestUtil.readFileInAST("resources/SimpleTestFile.java");
		DecoratorSelection ds = new DecoratorSelection();
		ds.addDecorator(new ProtectedDecorator());
		ds.addDecorator(new StyleDecorator());

		p.apply(ast);

		String expected = TestUtil.readFileInString("resources/result.xml");
		String actual = TestUtil.astToXMLString(ast, ds);
		assertEquals(expected, actual);

	}
}
