package org.deft.extension.persistence;

import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Set;

import org.deft.repository.ast.TreeNode;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

/**
 * Common base for modification and decoration tags.
 * 
 * @author Martin Heinzerling
 * 
 */
public abstract class Item {
	private Invoker invoker;
	private Map<String, String> params = new LinkedHashMap<String, String>();

	public Item(Invoker invoker) {
		this.invoker = invoker;
	}

	public Item(Invoker invoker, Map<String, String> params) {
		this.invoker = invoker;
		this.params = params;
	}

	public Item(Node node) {
		extractInvoker(node);
		extractParams(node);
	}

	private void extractParams(Node node) {
		try {
			NodeList nl = node.getChildNodes();

			for (int i = 0; i < nl.getLength(); i++) {
				Node n = nl.item(i);
				if (n.getNodeName() == "param") {
					String name = n.getAttributes().getNamedItem("name")
							.getTextContent();
					params.put(name, n.getTextContent().trim());
				}
			}
		} catch (Exception e) {
			throw new PersistenceException("Invalid parameter found: " + e);
		}
	}

	private void extractInvoker(Node node) {
		try {
			String className = node.getAttributes().getNamedItem("class")
					.getTextContent();
			invoker = (Invoker) Class.forName(className).newInstance();
		} catch (Exception e) {
			throw new PersistenceException("No valid invoker found: " + e);
		}
	}

	/**
	 * Apply stored modifictions/decoration to a AST.
	 */
	public void apply(TreeNode ast) {
		invoker.invoke(ast, params);
	}

	/**
	 * Creates a equivalent XML element .
	 * 
	 * @param doc
	 *            parent document to get all namespaces
	 * @return
	 */
	public Element toElement(Document doc) {
		Element root = createNode(doc);
		// Class
		root.setAttribute("class", invoker.getClass().getCanonicalName());
		Set<String> keys = params.keySet();
		for (String key : keys) {
			String value = params.get(key);
			Element p = doc.createElement("param");
			p.setAttribute("name", key);
			p.setTextContent(value);
			root.appendChild(p);
		}
		return root;
	}

	/**
	 * Returns the concrete type of the item as empty xml node.
	 */
	protected abstract Element createNode(Document doc);

	@Override
	public String toString() {
		return invoker + ":" + params.toString();
	}
}
