package org.deft.extension.persistence;

import java.io.File;
import java.io.FileOutputStream;
import java.util.LinkedList;
import java.util.List;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.transform.OutputKeys;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;

import org.deft.repository.ast.TreeNode;
import org.w3c.dom.Document;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

/**
 * Stores a list of modification and decoration to apply them to a AST.
 * 
 * @author Martin Heinzerling
 */
public class Persistence {

	private List<Modification> modifications = new LinkedList<Modification>();
	private List<Decoration> decorations = new LinkedList<Decoration>();

	public Persistence() {
	}

	public Persistence(File file) {
		load(file);
	}

	/**
	 * Load/initialize object from a file.
	 * 
	 * @param file
	 */
	public void load(File file) {
		DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
		DocumentBuilder builder;
		Document document;
		try {
			builder = factory.newDocumentBuilder();
			document = builder.parse(file);
		} catch (Exception e) {
			throw new PersistenceException(e.toString());
		}
		if (document.getDocumentElement().getNodeName() != "codesnippet") {
			throw new PersistenceException("Invalid file: codesnippet missing");
		}

		NodeList nl = document.getElementsByTagName("modification");
		for (int i = 0; i < nl.getLength(); i++) {
			Node n = nl.item(i);
			add(new Modification(n));
		}
		nl = document.getElementsByTagName("decoration");
		for (int i = 0; i < nl.getLength(); i++) {
			Node n = nl.item(i);
			add(new Decoration(n));
		}

	}

	/**
	 * Stores object to a file.
	 * 
	 * @param file
	 */
	public void store(File file) {
		try {
			DocumentBuilderFactory factory = DocumentBuilderFactory
					.newInstance();
			DocumentBuilder builder;
			builder = factory.newDocumentBuilder();
			Document document = builder.newDocument();

			Node root = document.createElement("codesnippet");
			document.appendChild(root);
			if (modifications.size() > 0) {
				Node m = document.createElement("modifications");
				root.appendChild(m);
				for (Modification mod : modifications) {
					m.appendChild(mod.toElement(document));
				}

			}
			if (decorations.size() > 0) {
				Node d = document.createElement("decorations");
				root.appendChild(d);
				for (Decoration dec : decorations) {
					d.appendChild(dec.toElement(document));
				}

			}

			Transformer transformer = TransformerFactory.newInstance()
					.newTransformer();
			transformer.setOutputProperty(OutputKeys.INDENT, "yes");
			transformer.setOutputProperty(
					"{http://xml.apache.org/xslt}indent-amount", "2");
			DOMSource source = new DOMSource(document);
			FileOutputStream os = new FileOutputStream(file);
			StreamResult result = new StreamResult(os);
			transformer.transform(source, result);
		} catch (Exception e) {
			throw new PersistenceException("Error while saving: " + e);
		}

	}

	/**
	 * 
	 * @param decoration
	 */
	public void add(Decoration decoration) {
		decorations.add(decoration);
	}

	/**
	 * 
	 * @param decoration
	 */
	public void remove(Decoration decoration) {
		decorations.remove(decoration);
	}

	/**
	 * 
	 * @param motification
	 */
	public void add(Modification motification) {
		modifications.add(motification);
	}

	/**
	 * 
	 * @param motification
	 */
	public void remove(Modification motification) {
		modifications.remove(motification);
	}

	/**
	 * Apply all modification/decations to a AST.
	 * 
	 * @param ast
	 */
	public void apply(TreeNode ast) {
		for (Modification m : modifications) {
			m.apply(ast);
		}
		for (Decoration d : decorations) {
			d.apply(ast);
		}
	}

	@Override
	public String toString() {
		return modifications.toString() + " -- " + decorations.toString();
	}
}
