package org.deft.extension.tools.astlayouter;

import java.util.HashMap;
import java.util.List;

import org.deft.repository.ast.Token;
import org.deft.repository.ast.TokenNode;

/**
 * Representation of the complete layout behind a AST.
 * 
 * @author Martin Heinzerling
 * 
 */
public class ASTLayout {

	private int standardLineBreakOnError = 2;
	private HashMap<Integer, TokenLine> map = new HashMap<Integer, TokenLine>();
	private int lastLineNumber;
	private int lineBreakWidth;

	public ASTLayout(List<TokenNode> tokenNodes) {

		reload(tokenNodes);

		calculateLineBreakWidth();
	}

	/**
	 * Reloads a AST (i.e. after modification) without recalculating the line
	 * break width.
	 * 
	 * @param tokenNodes
	 *            serialized AST
	 */
	public void reload(List<TokenNode> tokenNodes) {
		map.clear();
		Token lastToken = null;
		for (TokenNode tn : tokenNodes) {
			Token t = tn.getToken();
			int line = t.getLine();

			TokenLine tline = getOrCreateTokenList(line);
			tline.addToken(t);
			lastToken = t;
			lastLineNumber = line;
		}
		handleMultiLineLastToken(lastToken);

	}

	private void handleMultiLineLastToken(Token lastToken) {
		if (lastToken != null) {
			lastLineNumber += (lastToken.countLines() - 1);
		}
	}

	private TokenLine getOrCreateTokenList(Integer line) {
		TokenLine tline;
		if (map.containsKey(line)) {
			tline = map.get(line);
		} else {
			tline = new TokenLine(line);
			map.put(line, tline);
		}
		return tline;
	}

	/**
	 * @post linebreakwidth == 1 || linebreakwidth == 2
	 */
	// TODO O: Standard linebreakwidth
	private void calculateLineBreakWidth() {
		Integer startLine = 0;
		Integer firstValidLineNr = getNextValidLine(startLine);
		if (firstValidLineNr == null) {
			throw new RuntimeException("Couldn't calculate linebreak width");
		}

		startLine = firstValidLineNr + 1;
		Integer nextValidLineNr = getNextValidLine(startLine);
		if (nextValidLineNr == null) {
			lineBreakWidth = standardLineBreakOnError;
			return;
		}

		int endOffsetFirstLine = map.get(firstValidLineNr).getEndOffset();
		int startOffsetNextLine = map.get(nextValidLineNr).getOffset();

		lineBreakWidth = (startOffsetNextLine - endOffsetFirstLine)
				/ (nextValidLineNr - firstValidLineNr);

		if (lineBreakWidth != 1 && lineBreakWidth != 2) {

			lineBreakWidth = standardLineBreakOnError;

		}
	}

	// TODO I: Problem with nonvisible chars in "empty" lines. e.g. tab or
	// leading spaces
	/**
	 * Recalculates the offset of each token in the layout. Recommended after
	 * last operation.
	 */
	public void repairOffset() {
		int runnigOffset = 0;
		Integer linesAdded = 0;
		int skip = 0;
		int multilineEndCol = 0;
		for (int i = 1; i <= lastLineNumber; i += 1) {
			TokenLine tl = map.get(i);
			if (skip == 0 || tl != null) {
				if (skip == 0) {
					multilineEndCol = 0;
				}
				int[] arr = setOffsetAndGetLengthOfLine(tl, runnigOffset,
						multilineEndCol);
				int offset = arr[0];
				linesAdded = arr[1];
				skip = linesAdded - 1;
				if (skip > 1) {
					multilineEndCol = tl.getEndCol();
				}

				runnigOffset += offset;
			} else {
				skip--;
			}

		}

	}

	private int[] setOffsetAndGetLengthOfLine(TokenLine tl, int startOffset,
			int multilineEndCol) {
		if (tl == null) {
			return new int[] { lineBreakWidth, 1 };
		}

		int offset = 0;
		int lastProcessedCol = 1;
		if (multilineEndCol > 0) {
			offset = -lineBreakWidth;
			lastProcessedCol = multilineEndCol + 1;
		}

		int lastTokenLength = 0;
		int lines = 1;

		for (Token t : tl) {
			offset += t.getCol() - lastProcessedCol;
			t.setOffset(startOffset + offset);
			lastProcessedCol = t.getCol();
			lastTokenLength = t.getLength();
			lines = t.countLines();
		}

		return new int[] { offset + lastTokenLength + lineBreakWidth, lines };
	}

	/**
	 * 
	 * @param line
	 *            line index
	 * @return TokenLine
	 */
	public TokenLine getLine(int line) {
		return map.get(line);
	}

	/**
	 * 
	 * @param t
	 *            in line
	 * @return TokenLine
	 */
	public TokenLine getLine(Token t) {
		return map.get(t.getLine());
	}

	/**
	 * 
	 * 
	 * @param current
	 *            current line index
	 * @return index of next filled line.
	 */
	public Integer getNextValidLine(int current) {

		for (int i = current + 1; i <= lastLineNumber; i++) {
			if (map.containsKey(i)) {
				return i;
			}
		}
		return null;
	}

	/**
	 * 
	 * @param current
	 *            current line index
	 * @return index of the previous filled line (also multilines)
	 */
	public Integer getPreviousValidLine(int current) {
		for (int i = current - 1; i > 0; i--) {
			if (map.containsKey(i)) {
				// Endline is necessary for multiline tokens
				return map.get(i).getEndLine();
			}
		}
		return null;
	}

	/**
	 * 
	 * @param current
	 *            current line index
	 * @return index of the previous filled line (no multilines)
	 */
	public int getPreviousRealLine(int current) {
		for (int i = current - 1; i > 0; i--) {
			if (map.containsKey(i)) {
				return i;
			}
		}
		return 0;
	}

	/**
	 * Moves tokens by a given line offset.
	 * 
	 * @param token
	 *            Token
	 * @param offset
	 *            lines
	 * 
	 */
	public void moveLinesBeginningByToken(Token token, int offset) {
		TokenLine tl = getLine(token);
		int oldLineNr = tl.getLine();
		int newLineNr = oldLineNr + offset;
		TokenLine oldLine = new TokenLine(oldLineNr);
		TokenLine newLine = new TokenLine(newLineNr);
		// move lines in layouter
		boolean tokenFound = false;
		for (Token t : tl) {
			if (t == token) {
				tokenFound = true;
			}
			if (tokenFound) {
				t.moveLine(offset);
				newLine.addToken(t);
			} else {
				oldLine.addToken(t);
			}

		}

		map.put(oldLineNr, oldLine);
		moveLinesAfterLine(oldLineNr, 1);
		map.put(newLineNr, newLine);

	}

	/*
	 * TODO O: Zwischenspeichern und mit den Offset auf einmal anwenden. Durch
	 * nderung nur der Lines in der Hashmap auch Einsparung mglich.
	 */
	/**
	 * Moves tokens after a line by a given line offset.
	 * 
	 * @param targetLine
	 *            line index
	 * @param offset
	 *            lines
	 */
	public void moveLinesAfterLine(int targetLine, int offset) {
		if (offset > 0) {
			for (int i = lastLineNumber; i > targetLine; i--) {
				moveInTokenAndMap(i, offset);
			}
		} else {
			for (int i = targetLine + 1; i <= lastLineNumber; i++) {
				moveInTokenAndMap(i, offset);
			}
		}
		lastLineNumber += offset;
	}

	private void moveInTokenAndMap(int line, int offset) {
		if (offset == 0) {
			return;
		}
		TokenLine tokenInLine = map.get(line);
		if (tokenInLine != null) {
			tokenInLine.moveLine(offset);
			map.put(line + offset, tokenInLine);
			map.remove(line);
		}
	}

	/**
	 * Moves a line in horizontal direction. (Negative values allowed).
	 * 
	 * @param selectedLine
	 *            index
	 * @param offset
	 *            cols
	 */
	public void moveCol(int selectedLine, int offset) {
		TokenLine line = map.get(selectedLine);
		if (line == null || offset == 0) {
			return;
		}
		if (offset < 0) {
			int maxNegativOffset = -(line.getCol() - 1);
			if (offset < maxNegativOffset) {
				offset = maxNegativOffset;
			}
		}
		line.moveCol(0, offset);
	}

	/**
	 * 
	 * 
	 * @return last line in layout
	 */
	public int lastLine() {
		return lastLineNumber;
	}

	@Override
	public String toString() {
		return map.toString();
	}

}
