package org.deft.extension.tools.astlayouter;

import java.util.List;

import org.deft.repository.ast.Token;
import org.deft.repository.ast.TokenNode;

/**
 * The TokenLayouter provides basic layout operation on an AST.
 * 
 * @author Martin Heinzerling
 * 
 */
public class ASTLayouter {

	private ASTLayout layout;

	public ASTLayouter(List<TokenNode> tokenNodes) {
		layout = new ASTLayout(tokenNodes);

	}

	/**
	 * You have to reload a TokenLayouter after modification.
	 * 
	 * @param tokenNodes
	 *            serialized AST
	 */
	public void reload(List<TokenNode> tokenNodes) {
		layout.reload(tokenNodes);

	}

	/**
	 * Recalculates the offset of each token in the layout. Recommended after
	 * last operation.
	 */
	public void repairOffset() {
		layout.repairOffset();
	}

	/**
	 * 
	 * 
	 * @param targetToken
	 *            token
	 * @param offset
	 *            spaces
	 */
	public void insertSpacesBefore(Token targetToken, int offset) {
		// TODO O: Nicht zu weit nach Links schieben
		TokenLine tokensInSameLine = layout.getLine(targetToken);
		int targetTokenIndex = tokensInSameLine.indexOf(targetToken);
		int startIndex = targetTokenIndex;

		tokensInSameLine.moveCol(startIndex, offset);
	}

	/**
	 * 
	 * 
	 * @param targetToken
	 *            token
	 * @param offset
	 *            spaces
	 */
	public void insertSpacesAfter(Token targetToken, int offset) {
		throwExceptionOnNegative(offset);
		// TODO O: Negative Werte zulassen?
		TokenLine tokensInSameLine = layout.getLine(targetToken);
		int targetTokenIndex = tokensInSameLine.indexOf(targetToken);
		int startIndex = targetTokenIndex + 1;

		if (targetToken.isSingleLine()) {
			tokensInSameLine.moveCol(startIndex, offset);
		} else {
			// Wenn nach Multilinetoken noch etwa steht!
			TokenLine endline = layout.getLine(targetToken.getEndLine());
			if (endline != null) {
				endline.moveCol(0, offset);
			}
		}
	}

	/**
	 * 
	 * 
	 * @param targetToken
	 *            token
	 * @param offset
	 *            lines
	 */
	public void insertLinesBefore(Token targetToken, int offset) {
		throwExceptionOnNegative(offset);
		Integer selectedLine = targetToken.getLine();
		selectedLine--;

		layout.moveLinesAfterLine(selectedLine, offset);
	}

	/**
	 * Insert lines before the first token.
	 * 
	 * @param offset
	 *            lines
	 */
	public void insertLinesBefore(int offset) {
		int firstline = layout.getNextValidLine(0);
		Token targetToken = layout.getLine(firstline).getFirstToken();
		insertLinesBefore(targetToken, offset);
	}

	/**
	 * 
	 * @param targetToken
	 *            token
	 * @param offset
	 *            lines
	 */
	public void insertLinesAfter(Token targetToken, int offset) {
		throwExceptionOnNegative(offset);
		Integer selectedLine = targetToken.getLine();

		layout.moveLinesAfterLine(selectedLine, offset);
	}

	/**
	 * 
	 * @param targetToken
	 *            token
	 */
	public void insertLineBreakBefore(Token targetToken) {
		layout.moveLinesBeginningByToken(targetToken, 1);
	}

	/**
	 * 
	 * @param targetToken
	 *            token
	 */
	public void insertLineBreakAfter(Token targetToken) {
		if (targetToken == layout.getLine(targetToken).getLastToken()) {
			insertLinesAfter(targetToken, 1);
		} else {
			Token nextToken = layout.getLine(targetToken).getNextToken(
					targetToken);
			layout.moveLinesBeginningByToken(nextToken, 1);
		}
	}

	/**
	 * Removes linebreak in the line of the Token.
	 * 
	 * @param targetToken
	 *            token
	 */
	public void removeLineBreakAfter(Token targetToken) {
		TokenLine line = layout.getLine(targetToken);
		int eol = line.getLastToken().getEndCol();

		TokenLine nextLine = layout.getLine(line.getEndLine() + 1);
		if (nextLine != null) {
			Token firstTokenOfNextLine = nextLine.getFirstToken();

			int offset = eol - firstTokenOfNextLine.getCol();
			insertSpacesBefore(firstTokenOfNextLine, offset);
		}
		layout.moveLinesAfterLine(targetToken.getLine(), -1);

	}

	/**
	 * Removes linebreak in the line of the Token.
	 * 
	 * @param targetToken
	 *            token
	 */
	public void removeLineBreakBefore(Token targetToken) {
		int line = layout.getPreviousRealLine(targetToken.getLine());
		if (line == 0) {
			return;
		}
		removeLineBreakAfter(layout.getLine(line).getFirstToken());

	}

	private void throwExceptionOnNegative(int i) {
		if (i < 0) {
			throw new InvalidArgumentException("Offset have to be non-negative");
		}
	}

	// TODO O: Geht sicher auch schneller
	/**
	 * @param targetToken
	 *            token
	 */
	public void trim(Token targetToken) {
		trimLeft(targetToken);
		trimRight(targetToken);
	}

	/**
	 * 
	 * @param targetToken
	 *            token
	 */
	public void trimLeft(Token targetToken) {
		TokenLine tokensInSameLine = layout.getLine(targetToken);
		int targetTokensIndex = tokensInSameLine.indexOf(targetToken);

		if (targetTokensIndex == 0) {
			// targetToken ist erster Token
			int offset = targetToken.getCol() - 1;
			tokensInSameLine.moveCol(0, -offset);
		} else {
			// Vorgnger rechts trimmen
			trimRight(tokensInSameLine.getPreviousToken(targetToken));
		}

	}

	/**
	 * 
	 * @param targetToken
	 *            token
	 */
	public void trimRight(Token targetToken) {
		TokenLine tokensInSameLine = layout.getLine(targetToken);
		Token rightNeighbourToken = tokensInSameLine.getNextToken(targetToken);
		if (rightNeighbourToken == null) {
			return;
		}
		int rightNeighbourIndex = tokensInSameLine.indexOf(rightNeighbourToken);

		int oldCol = rightNeighbourToken.getCol();
		int newCol = targetToken.getCol()
				+ targetToken.getText().trim().length();
		int offset = newCol - oldCol;
		tokensInSameLine.moveCol(rightNeighbourIndex, offset);

	}

	/**
	 * 
	 * @param targetToken
	 *            token
	 */
	public void trimLinesBefore(Token targetToken) {
		int targetLine = targetToken.getLine();
		Integer prevLine = layout.getPreviousValidLine(targetLine);
		if (prevLine == null) {
			return; // no empty lines
		}
		int offset = targetLine - prevLine - 1;
		layout.moveLinesAfterLine(prevLine, -offset);
	}

	/**
	 * 
	 * @param targetToken
	 *            token
	 */
	public void trimLinesAfter(Token targetToken) {
		int targetLine = targetToken.getEndLine();
		Integer nextLine = layout.getNextValidLine(targetLine);
		if (nextLine == null) {
			return; // no empty lines
		}
		int offset = nextLine - targetLine - 1;
		layout.moveLinesAfterLine(targetLine, -offset);
	}

	/**
	 * Indents a block of token.
	 * 
	 * @param startToken
	 *            token
	 * @param endToken
	 *            inclusive
	 * @param offset
	 *            cols
	 */
	public void indentRelative(Token startToken, Token endToken, int offset) {
		int startLine = startToken.getLine();
		int endLine = endToken.getLine();
		moveLines(offset, startLine, endLine);
	}

	private void moveLines(int offset, int startLine, int endLine) {
		for (int i = startLine; i <= endLine; i++) {
			layout.moveCol(i, offset);
		}
	}

	/**
	 * Indents the whole AST.
	 * 
	 * @param offset
	 *            cols
	 */
	public void indentRelative(int offset) {
		int endLine = layout.lastLine();
		moveLines(offset, 0, endLine);
	}

	@Override
	public String toString() {
		return layout.toString();
	}

	/**
	 * First token starts at position (1,1).
	 * 
	 */
	public void reset() {
		int firstLine = layout.getNextValidLine(0);
		int firstCol = layout.getLine(firstLine).getCol();
		layout.moveLinesAfterLine(0, -(firstLine - 1));
		indentRelative(-(firstCol - 1));
	}

	public int getEndLine() {

		return layout.lastLine();
	}

	/**
	 * 
	 * @param targetToken
	 *            token
	 * @return start of line with this token
	 */
	public int getColOfLine(Token targetToken) {
		return layout.getLine(targetToken).getCol();
	}

	/**
	 * 
	 * @return last col of the layout
	 */
	public int getEndCol() {
		TokenLine tl = layout.getLine(getEndLine());
		if (tl == null) {
			int lastValidLine = layout.getPreviousRealLine(getEndLine());
			tl = layout.getLine(lastValidLine);
		}
		return tl.getEndCol();
	}

	/**
	 * 
	 * @param targetToken
	 *            token
	 * @return Token or null
	 */
	public Token getNextTokenInSameLine(Token targetToken) {
		int line = targetToken.getLine();
		TokenLine tl = layout.getLine(line);
		Token next = tl.getNextToken(targetToken);
		return next;

	}

	/**
	 * 
	 * @param targetToken
	 *            token
	 * @return Token or null
	 */
	public Token getPrevTokenInSameLine(Token targetToken) {
		int line = targetToken.getLine();
		TokenLine tl = layout.getLine(line);
		Token prev = tl.getPreviousToken(targetToken);
		return prev;
	}

}
