package org.deft.extension.tools.astlayouter;

import java.util.Map;

import org.deft.extension.persistence.Invoker;
import org.deft.extension.persistence.PersistenceException;
import org.deft.repository.ast.Token;
import org.deft.repository.ast.TreeNode;

/**
 * 
 * Handles a invoker call from a Persistence object and applies a layout change
 * to a XPath. <br>
 * Parameter<br>
 * <br>
 * method - insertSpacesBefore, insertSpacesAfter, insertLinesBefore,
 * insertLinesAfter, insertLineBreakBefore, insert- LineBreakAfter,
 * removeLineBreakAfter, removeLineBreakBefore, trim, trimLeft, trimRight,
 * trimLinesBefore, trimLinesAfter or indentRelative<br>
 * <br>
 * offset - int; necessary with insertSpacesBefore, insertSpacesAfter,
 * insertLinesBefore, insertLinesAfter and indentRelative<br>
 * <br>
 * location - XPath; necessary with all but indentRelative<br>
 * <br>
 * startlocation - XPath; necessary with all but indentRelative<br>
 * <br>
 * endlocation - XPath; necessary with indentRelative<br>
 * <br>
 * 
 * @author Martin Heinzerling
 * 
 */
public class ASTLayouterInvoker implements Invoker {

	private String method;
	private int offset = 0;
	private Token target = null;

	@Override
	public void invoke(TreeNode ast, Map<String, String> params) {
		readMethodParam(params);
		readOffsetParam(params);
		readLocationParam(ast, params);

		ASTLayouter layouter = new ASTLayouter(ast.serialize());

		invokeMethod(ast, params, layouter);
		layouter.repairOffset();

	}

	private void invokeMethod(TreeNode ast, Map<String, String> params,
			ASTLayouter layouter) {
		if (method.startsWith("insert")) {
			invokeInsertMethod(layouter);
		} else if (method.equals("removeLineBreakAfter")) {
			layouter.removeLineBreakAfter(target);
		} else if (method.equals("removeLineBreakBefore")) {
			layouter.removeLineBreakBefore(target);
		} else if (method.equals("trim")) {
			layouter.trim(target);
		} else if (method.equals("trimLeft")) {
			layouter.trimLeft(target);
		} else if (method.equals("trimRight")) {
			layouter.trimRight(target);
		} else if (method.equals("trimLinesBefore")) {
			layouter.trimLinesBefore(target);
		} else if (method.equals("trimLinesAfter")) {
			layouter.trimLinesAfter(target);
		} else if (method.equals("indentRelative")) {
			invokeIndent(ast, layouter, params, offset);
		} else {
			throw new PersistenceException("Unknown method: '" + method + "'");
		}
	}

	private void invokeInsertMethod(ASTLayouter layouter) {
		if (method.equals("insertSpacesBefore")) {
			layouter.insertSpacesBefore(target, offset);
		} else if (method.equals("insertSpacesAfter")) {
			layouter.insertSpacesAfter(target, offset);
		} else if (method.equals("insertLinesBefore")) {
			layouter.insertLinesBefore(target, offset);
		} else if (method.equals("insertLinesAfter")) {
			layouter.insertLinesAfter(target, offset);
		} else if (method.equals("insertLineBreakBefore")) {
			layouter.insertLineBreakBefore(target);
		} else if (method.equals("insertLineBreakAfter")) {
			layouter.insertLineBreakAfter(target);
		}
	}

	private void readLocationParam(TreeNode ast, Map<String, String> params) {
		if (!method.equals("indentRelative")) {

			String location = params.get("location");
			if (location == null) {
				throw new PersistenceException("Parameter 'location' required");
			}
			try {
				if (method.endsWith("After") || method.endsWith("Right")) {
					target = ast.executeXPathQuery(location).get(0)
							.getLastToken();
				} else {
					target = ast.executeXPathQuery(location).get(0)
							.getFirstToken();
				}
			} catch (NullPointerException e) {
				throw new PersistenceException("Invalid 'location'");
			}

		}
	}

	private void readOffsetParam(Map<String, String> params) {
		if (offsetRequired(method)) {
			String offsetString = params.get("offset");
			if (offsetString == null) {
				throw new PersistenceException("Parameter 'offset' required");
			}
			offset = new Integer(offsetString);
		}
	}

	private void readMethodParam(Map<String, String> params) {
		method = params.get("method");
		if (method == null) {
			throw new PersistenceException("Parameter 'method' required");
		}
	}

	private void invokeIndent(TreeNode ast, ASTLayouter layouter,
			Map<String, String> params, int offset) {
		String startlocation = params.get("startlocation");
		String endlocation = params.get("endlocation");
		if (startlocation == null || endlocation == null) {
			throw new PersistenceException(
					"Parameter 'startlocation' and 'endlocation' required");
		}
		Token startToken, endToken;
		try {
			startToken = ast.executeXPathQuery(startlocation).get(0)
					.getFirstToken();
			endToken = ast.executeXPathQuery(endlocation).get(0).getLastToken();
		} catch (NullPointerException e) {
			throw new PersistenceException("Invalid start or end location");
		}
		layouter.indentRelative(startToken, endToken, offset);
	}

	private boolean offsetRequired(String method) {
		return method.startsWith("insertSpaces")
				|| method.startsWith("insertLines")
				|| method.equals("indentRelative");
	}
}
