package org.deft.extension.tools.astlayouter;

import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;

import org.deft.repository.ast.Token;

/**
 * Representation of tokens of the AST in one line.
 * 
 * @author Martin Heinzerling
 * 
 */
public class TokenLine implements Iterable<Token> {

	private int line;
	private List<Token> list = new LinkedList<Token>();

	public TokenLine(int line) {
		this.line = line;
	}

	/**
	 * 
	 * @param t
	 *            Token
	 */
	public void addToken(Token t) {
		list.add(t);
	}

	/**
	 * Relative move in a line, starting at a given token index.
	 * 
	 * @param startIndex
	 *            start token index
	 * @param offset
	 *            relative movement
	 */
	public void moveCol(int startIndex, int offset) {
		for (int i = startIndex, size = list.size(); i < size; i++) {
			Token t = list.get(i);
			t.moveCol(offset);
		}
	}

	/**
	 * Relative move of the whole line.
	 * 
	 * @param offset
	 *            relative movement
	 */
	public void moveLine(int offset) {

		for (Token t : list) {
			t.moveLine(offset);
		}
		this.line += offset;
	}

	@Override
	public Iterator<Token> iterator() {
		return list.iterator();
	}

	/**
	 * 
	 * @param t
	 *            Token
	 * @return index
	 */
	public int indexOf(Token t) {

		return list.indexOf(t);
	}

	/**
	 * 
	 * @param i
	 *            index
	 * @return Token
	 */
	public Token getTokenAtIndex(int i) {
		return list.get(i);
	}

	/**
	 * 
	 * @return Token or null
	 */
	public Token getFirstToken() {
		if (list.size() == 0) {
			return null;
		}
		return list.get(0);

	}

	public Token getLastToken() {
		return list.get(list.size() - 1);
	}

	/**
	 * Not for iteration!!!!
	 * 
	 * @param t
	 *            Token
	 * @return Token or null
	 */
	public Token getPreviousToken(Token t) {
		int index = list.indexOf(t);
		if (index == 0) {
			return null;
		}
		return list.get(index - 1);
	}

	/**
	 * Not for iteration!!!!
	 * 
	 * @param t
	 *            Token
	 * @return Token or null
	 */
	public Token getNextToken(Token t) {
		int index = list.indexOf(t);
		if (index == list.size() - 1) {
			return null;
		}
		return list.get(index + 1);
	}

	public int getLine() {
		return line;
	}

	public int getCol() {
		return list.get(0).getCol();
	}

	/**
	 * 
	 * @return First offset of line
	 */
	public int getOffset() {
		Token firstTokenOfLine = getFirstToken();
		return firstTokenOfLine.getOffset();
	}

	public int getEndLine() {
		return line + list.get(list.size() - 1).countLines() - 1;
	}

	public int getEndCol() {
		return list.get(list.size() - 1).getEndCol();
	}

	/**
	 * 
	 * @return EOL offset
	 */
	public int getEndOffset() {
		Token lastTokenOfLine = getLastToken();
		return getLastToken().getOffset() + lastTokenOfLine.getLength();
	}

	/**
	 * 
	 * @return tokens in line
	 */
	public int size() {
		return list.size();
	}

	@Override
	public String toString() {
		return list.toString();
	}
}
