package org.deft.extension.tools.astmodifier;

import java.util.LinkedList;
import java.util.List;

import org.deft.extension.XPath;
import org.deft.extension.decoration.modify.ModifiedInformation;
import org.deft.extension.tools.astlayouter.ASTLayouter;
import org.deft.extension.tools.astmodifier.comment.CommentTypeVisitor;
import org.deft.extension.tools.astmodifier.syntax.SyntaxCheckerVisitor;
import org.deft.repository.ast.Token;
import org.deft.repository.ast.TreeNode;

/**
 * Provides modification operation on a AST.
 * 
 * @author Martin Heinzerling
 */
public class ASTModifier {

	private SyntaxCheckerVisitor syntaxChecker;

	private TreeNode parent;
	private ASTLayouter tli;
	private ASTLayouter tla;

	public ASTModifier(SyntaxCheckerVisitor sc) {
		syntaxChecker = sc;
	}

	private TreeNode getFirstTarget(TreeNode ast, XPath location) {
		for (TreeNode target : ast.executeXPathQuery(location.toString())) {
			return target;
		}
		return null;
	}

	/**
	 * 
	 * @param location
	 *            to insert the treenode
	 */
	public void appendAfter(TreeNode ast, TreeNode insert, XPath location,
			int emptyLinesBefore, int emptyLinesAfter, int spacesBefore,
			int spacesAfter) {
		tla = new ASTLayouter(ast.serialize());

		TreeNode target = getFirstTarget(ast, location);

		int insertAtIndex = target.getSiblingIndex() + 1;
		Token targetToken = target.getLastToken();
		Token nextToken = tla.getNextTokenInSameLine(targetToken);
		int startCol = tla.getColOfLine(targetToken);
		int targetCol = targetToken.getEndCol();

		initTokenLayouterAndInsertInAST(target, insert, insertAtIndex);

		insertEmptyLinesAfter(targetToken, emptyLinesBefore, emptyLinesAfter,
				target, 1);

		indentInsertion(spacesBefore, emptyLinesBefore, targetCol, startCol);

		insertWhiteSpaceAfter(nextToken, startCol, emptyLinesAfter, spacesAfter);

		ast.invalidateCache();
		tla.reload(ast.serialize()); // TODO O: Neueinlesen der Tokenliste
		tla.repairOffset();

	}

	/**
	 * 
	 * @param location
	 *            to insert the treenode
	 */
	public void appendBefore(TreeNode ast, TreeNode insert, XPath location,
			int emptyLinesBefore, int emptyLinesAfter, int spacesBefore,
			int spacesAfter) {
		tla = new ASTLayouter(ast.serialize());

		TreeNode target = getFirstTarget(ast, location);

		int insertAtIndex = target.getSiblingIndex();
		Token targetToken = target.getFirstToken();
		Token prevToken = tla.getPrevTokenInSameLine(targetToken);
		int startCol = tla.getColOfLine(targetToken);
		int targetCol = 1;
		if (prevToken != null) {
			targetCol = prevToken.getEndCol();
		}

		initTokenLayouterAndInsertInAST(target, insert, insertAtIndex);
		insertEmptyLinesBefore(targetToken, emptyLinesBefore, emptyLinesAfter,
				target);
		indentInsertion(spacesBefore, emptyLinesBefore, targetCol, startCol);
		insertWhiteSpaceAfter(targetToken, startCol, emptyLinesAfter,
				spacesAfter);

		ast.invalidateCache();
		tla.reload(ast.serialize()); // TODO O: Neueinlesen der Tokenliste

		tla.repairOffset();
	}

	/**
	 * @param location
	 *            to replace the treenode
	 */
	public void replace(TreeNode ast, TreeNode insert, XPath location,
			int emptyLinesBefore, int emptyLinesAfter, int spacesBefore,
			int spacesAfter) {
		tla = new ASTLayouter(ast.serialize());

		TreeNode target = getFirstTarget(ast, location);

		int insertAtIndex = target.getSiblingIndex() + 1;
		Token targetToken = target.getLastToken();
		Token nextToken = tla.getNextTokenInSameLine(targetToken);
		int startCol = tla.getColOfLine(targetToken);
		int targetCol = targetToken.getCol();
		int replacedLines = target.getEndLine() - target.getStartLine() + 1;

		initTokenLayouterAndInsertInAST(target, insert, insertAtIndex);
		insertEmptyLinesAfter(targetToken, emptyLinesBefore, emptyLinesAfter,
				target, replacedLines);
		indentInsertion(spacesBefore, emptyLinesBefore, targetCol, startCol);
		insertWhiteSpaceAfter(nextToken, startCol, emptyLinesAfter, spacesAfter);

		parent.removeChild(target);
		ast.invalidateCache();
		tla.reload(ast.serialize()); // TODO O: Neueinlesen der Tokenliste

		tla.repairOffset();
	}

	/**
	 * 
	 * @param location
	 *            to replace the treenode
	 * @param width
	 *            -1 all following sibling
	 */
	public void replaceList(TreeNode ast, TreeNode insert, XPath location,
			int width, int emptyLinesBefore, int emptyLinesAfter,
			int spacesBefore, int spacesAfter) {

		// delete other
		TreeNode target = getFirstTarget(ast, location);
		TreeNode remove = target.getNextSibling();
		int c = 0;
		while (remove != null && (width == -1 || c < width)) {
			TreeNode temp = remove;
			remove = remove.getNextSibling();
			temp.releaseFromParent();
			c++;

		}

		// replace first
		replace(ast, insert, location, emptyLinesBefore, emptyLinesAfter,
				spacesBefore, spacesAfter);
	}

	private void checkSyntax(TreeNode parent, TreeNode insert, int insertAt) {
		if (!syntaxChecker.check(parent, insert, insertAt)) {
			throw new ModifierException(insert.toString()
					+ "not allowed under " + parent.toString());
		}
	}

	private void initTokenLayouterAndInsertInAST(TreeNode target,
			TreeNode insert, int insertAtIndex) {
		parent = target.getParent();
		checkSyntax(parent, insert, insertAtIndex);
		insert.addInformation(new ModifiedInformation());
		tli = new ASTLayouter(insert.serialize());
		tli.reset();
		parent.addChild(insertAtIndex, insert);
	}

	private void insertEmptyLinesAfter(Token targetToken, int emptyLinesBefore,
			int emptyLinesAfter, TreeNode target, int replacedLines) {
		int linesToInsert = tli.getEndLine() + emptyLinesBefore
				+ emptyLinesAfter - 1; // eine Zeile darf es sein

		if (linesToInsert > 0) {
			tla.insertLineBreakAfter(targetToken);
			tla.insertLinesAfter(targetToken, linesToInsert - 1);
			// minus den Linebreak
		}
		tli.insertLinesBefore(target.getEndLine() - replacedLines
				+ emptyLinesBefore);
	}

	private void insertEmptyLinesBefore(Token targetToken,
			int emptyLinesBefore, int emptyLinesAfter, TreeNode target) {
		int linesToInsert = tli.getEndLine() + emptyLinesBefore
				+ emptyLinesAfter - 1; // eine Zeile darf es sein

		tli.insertLinesBefore(target.getStartLine() - 1 + emptyLinesBefore);

		if (linesToInsert > 0) {
			tla.insertLineBreakBefore(targetToken);
			tla.insertLinesBefore(targetToken, linesToInsert - 1);
			// minus den Linebreak
		}
	}

	private void insertWhiteSpaceAfter(Token nextToken, int startCol,
			int emptyLinesAfter, int spacesAfter) {
		if (nextToken != null) {
			int tcol;
			int ccol = nextToken.getCol();

			if (emptyLinesAfter == 0) {
				tcol = tli.getEndCol() + spacesAfter;

			} else {
				tcol = startCol + spacesAfter;
			}
			int offset = tcol - ccol;
			tla.insertSpacesBefore(nextToken, offset);
		}
	}

	private void indentInsertion(int spacesBefore, int emptyLinesBefore,
			int targetCol, int startCol) {
		int baseIndent;
		if (emptyLinesBefore == 0) {
			baseIndent = targetCol - 1;
		} else {
			baseIndent = startCol - 1;
		}
		tli.indentRelative(baseIndent + spacesBefore);
	}

	/**
	 * @param location
	 *            to replace
	 * @param width
	 *            -1 all following sibling
	 */
	public void changeToComment(TreeNode ast, XPath location, int width,
			CommentTypeVisitor v) {
		List<TreeNode> nodes = new LinkedList<TreeNode>();

		TreeNode target = getFirstTarget(ast, location);
		if (target == null) {
			return;
		}
		nodes.add(target);

		TreeNode next = target.getNextSibling();
		int c = 0;
		while (next != null && (width == -1 || c < width)) {
			nodes.add(next);
			next = next.getNextSibling();
			c++;
		}
		List<TreeNode> comments = v.execute(nodes);

		TreeNode first = comments.get(0);

		replaceList(ast, first, location, width, 0, 0, 0, 0);

		for (int i = comments.size() - 1; i > 0; i--) {
			appendAfter(ast, comments.get(i), new XPath(first.getXpath()), 1,
					0, 0, 0);
		}
	}
}
