package org.deft.extension.tools.astmodifier;

import java.util.Map;

import org.deft.extension.XPath;
import org.deft.extension.persistence.Invoker;
import org.deft.extension.persistence.PersistenceException;
import org.deft.extension.tools.astmodifier.comment.CommentTypeVisitor;
import org.deft.extension.tools.astmodifier.syntax.NoSyntaxCheckerVisitor;
import org.deft.extension.tools.astmodifier.syntax.SyntaxCheckerVisitor;
import org.deft.language.java.parser.JavaParser;
import org.deft.repository.ast.TreeNode;

/**
 * 
 * <br>
 * method - appendAfter, appendBefore, replace, replaceList and changeToComment<br>
 * location - XPath syntaxchecker - SyntaxCheckerVisitor; optional
 * (NoSyntaxCheckerVisitor)<br>
 * insert - String;
 * 
 * <pre>
 * FILE+||+XPATH
 * </pre>
 * 
 * or
 * 
 * <pre>
 * (K_CLASS_BODY_DECLARATIONS |
 * K_COMPILATION_UNIT | K_EXPRESSION | K_STATEMENTS)+
 * ||+SOURCECODE|FILE+||+XPATH
 * </pre>
 * 
 * z.B. K_EXPRESSION||x=3||/Expression; necessary with all but changeToComment<br>
 * emptyLinesBefore - int; optional (0) necessary with all but changeToComment<br>
 * emptyLinesAfter - int; optional (0) necessary with all but changeToComment <br>
 * spacesBefore - int; optional (0) necessary with all but changeToComment <br>
 * spacesAfter - int; optional (0) necessary with all but changeToComment <br>
 * width - int; optional (0) only with changeToComment and replaceList<br>
 * commentType - CommentTypeVisitor; only with changeToComment
 * 
 * @author Martin Heinzerling
 * 
 */
public class ASTModifierInvoker implements Invoker {
	private String method;
	private SyntaxCheckerVisitor scv = null;
	private XPath location;
	private TreeNode insert = null;
	private int emptyLinesBefore = 0, emptyLinesAfter = 0, spacesBefore = 0,
			spacesAfter = 0, width = 0;

	@Override
	public void invoke(TreeNode ast, Map<String, String> params) {
		readMethod(params);
		readLocation(params);
		readChecker(params);
		readNumbers(params);
		readInsert(params);

		ASTModifier modifier = new ASTModifier(scv);

		invokeMethod(ast, modifier, params);
	}

	private void readInsert(Map<String, String> params) {
		if (method.startsWith("append") || method.startsWith("replace")) {
			insert = makeTreeNode(params.get("insert"));
			if (insert.getName().equals("root")) {
				throw new ModifierException(
						"It is not possible to insert a root node");
			}
		}
	}

	private void readChecker(Map<String, String> params) {
		try {
			if (params.get("syntaxchecker") != null) {
				scv = (SyntaxCheckerVisitor) Class.forName(
						params.get("syntaxchecker")).newInstance();
			} else {
				scv = new NoSyntaxCheckerVisitor();
			}
		} catch (Exception e) {
			throw new PersistenceException("Invalid syntaxchecker");
		}
	}

	private void readLocation(Map<String, String> params) {
		String l = params.get("location");
		if (l == null) {
			throw new PersistenceException("Parameter 'location' required");
		}
		location = new XPath(l);
	}

	private void readMethod(Map<String, String> params) {
		method = params.get("method");
		if (method == null) {
			throw new PersistenceException("Parameter 'method' required");
		}
	}

	private void invokeMethod(TreeNode ast, ASTModifier modifier,
			Map<String, String> params) {
		if (method.equals("appendAfter")) {
			modifier.appendAfter(ast, insert, location, emptyLinesBefore,
					emptyLinesAfter, spacesBefore, spacesAfter);
		} else if (method.equals("appendBefore")) {
			modifier.appendBefore(ast, insert, location, emptyLinesBefore,
					emptyLinesAfter, spacesBefore, spacesAfter);
		} else if (method.equals("replace")) {
			modifier.replace(ast, insert, location, emptyLinesBefore,
					emptyLinesAfter, spacesBefore, spacesAfter);
		} else if (method.equals("replaceList")) {
			modifier.replaceList(ast, insert, location, width,
					emptyLinesBefore, emptyLinesAfter, spacesBefore,
					spacesAfter);
		} else if (method.equals("changeToComment")) {
			invokeComment(ast, modifier, params);
		} else {
			throw new PersistenceException("Unknown method: '" + method + "'");
		}
	}

	private void invokeComment(TreeNode ast, ASTModifier modifier,
			Map<String, String> params) {
		CommentTypeVisitor v = null;
		if (params.get("commentType") != null) {
			try {
				v = (CommentTypeVisitor) Class.forName(
						params.get("commentType")).newInstance();
			} catch (Exception e) {
				throw new PersistenceException("Invalid commentType: " + e);
			}
		} else {
			throw new PersistenceException("Parameter 'commentType' required");
		}

		modifier.changeToComment(ast, location, width, v);
	}

	private void readNumbers(Map<String, String> params) {
		if (params.get("emptyLinesBefore") != null) {
			emptyLinesBefore = new Integer(params.get("emptyLinesBefore"));
		}
		if (params.get("emptyLinesAfter") != null) {
			emptyLinesAfter = new Integer(params.get("emptyLinesAfter"));
		}
		if (params.get("spacesBefore") != null) {
			spacesBefore = new Integer(params.get("spacesBefore"));
		}
		if (params.get("spacesAfter") != null) {
			spacesAfter = new Integer(params.get("spacesAfter"));
		}
		if (params.get("width") != null) {
			width = new Integer(params.get("width"));
		}
	}

	private TreeNode makeTreeNode(String string) {
		if (string == null) {
			throw new PersistenceException("Parameter 'insert' required");
		}

		return JavaParser.get(string);
	}
}
