package org.deft.extension.tools.tokentrimmer;

import java.io.File;
import java.util.HashMap;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;

import org.deft.extension.tools.astlayouter.ASTLayouter;
import org.deft.repository.ast.TreeNode;
import org.w3c.dom.Document;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

/**
 * Remove leading and trailing whitespace defined by a language dependent
 * specification.
 * 
 * @author Martin Heinzerling
 */
public class TokenAutoTrimmer {

	/**
	 * Possible direction to trim a token.
	 */
	private static enum TrimDirection {
		NONE, LEFT, RIGHT, BEFORE, AFTER, BOTH
	};

	/**
	 * Pair of horizontal an vertical trim direction.
	 */
	private static final class TrimPair {
		private TrimDirection h = TrimDirection.NONE;
		private TrimDirection v = TrimDirection.NONE;

		private TrimPair() {

		}
	};

	private static HashMap<String, HashMap<String, TrimPair>> trimMaps = new HashMap<String, HashMap<String, TrimPair>>();
	private static String currentTrimMapID = null;

	/**
	 * Creates a specific autotrimmer and connect it with an id.
	 * 
	 */
	public static void create(String id, String trimerDescriptionFile) {
		DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
		DocumentBuilder builder;
		Document document;
		try {
			builder = factory.newDocumentBuilder();
			document = builder.parse(new File(trimerDescriptionFile));
		} catch (Exception e) {
			throw new TrimmerException(e.toString());
		}
		trimMaps.put(id, load(document));
		currentTrimMapID = id;
	}

	/**
	 * Sets a previously create autotrimmer to the new standard one.
	 * 
	 * @param id
	 */
	public static void set(String id) {
		if (trimMaps.containsKey(id)) {
			currentTrimMapID = id;
		} else {
			throw new TrimmerException("Trimming " + id + " unknown");
		}
	}

	private static HashMap<String, TrimPair> load(Document document) {
		HashMap<String, TrimPair> trimMap = new HashMap<String, TrimPair>();
		NodeList nl = document.getElementsByTagName("trim");

		for (int i = 0; i < nl.getLength(); i++) {
			Node n = nl.item(i);
			TrimPair tp = getTrimAttributes(n);
			String content = n.getTextContent();
			String[] tokens = content.split(",");
			for (int c = 0, s = tokens.length; c < s; c++) {
				trimMap.put(tokens[c].trim(), tp);
			}
		}
		return trimMap;
	}

	private static TrimPair getTrimAttributes(Node n) {
		TrimPair tp = new TrimPair();
		Node v = n.getAttributes().getNamedItem("v");
		if (v == null) {
			tp.v = TrimDirection.NONE;
		} else {
			tp.v = toEnum(v.getNodeValue());
		}

		Node h = n.getAttributes().getNamedItem("h");
		if (h == null) {
			tp.h = TrimDirection.NONE;
		} else {
			tp.h = toEnum(h.getNodeValue());
		}
		return tp;
	}

	private static TrimDirection toEnum(String nodeValue) {
		if (nodeValue.equals("left")) {
			return TrimDirection.LEFT;
		}
		if (nodeValue.equals("right")) {
			return TrimDirection.RIGHT;
		}
		if (nodeValue.equals("before")) {
			return TrimDirection.BEFORE;
		}
		if (nodeValue.equals("after")) {
			return TrimDirection.AFTER;
		}
		if (nodeValue.equals("both")) {
			return TrimDirection.BOTH;
		}
		return TrimDirection.NONE;
	}

	private ASTLayouter tl;

	public TokenAutoTrimmer(TreeNode root) {

		tl = new ASTLayouter(root.serialize());

	}

	/**
	 * Trims a treenode by the current active autotrim definition.
	 * 
	 * @param tn
	 */
	public void trim(TreeNode tn) {

		if (currentTrimMapID == null) {
			return;
		}
		HashMap<String, TrimPair> trimMap = trimMaps.get(currentTrimMapID);
		TrimPair tp = trimMap.get(tn.getName());

		if (tp == null) {

			return;
		}
		switch (tp.h) {
		case LEFT:
			tl.trimLeft(tn.getFirstToken());
			break;
		case RIGHT:
			tl.trimRight(tn.getLastToken());
			break;
		case BOTH:
			tl.trimLeft(tn.getFirstToken());
			tl.trimRight(tn.getLastToken());
			break;
		default:
			break;
		}
		switch (tp.v) {
		case BEFORE:
			tl.trimLinesBefore(tn.getFirstToken());
			break;
		case AFTER:
			tl.trimLinesAfter(tn.getLastToken());
			break;
		case BOTH:
			tl.trimLinesBefore(tn.getFirstToken());
			tl.trimLinesAfter(tn.getLastToken());
			break;
		default:
			break;
		}
	}
}
