package org.deft.language.java.parser;

import java.io.BufferedReader;
import java.io.CharArrayReader;
import java.io.IOException;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.Stack;

import org.deft.repository.ast.Position;
import org.deft.repository.ast.Token;
import org.deft.repository.ast.TokenNode;
import org.deft.repository.ast.TreeNode;
import org.deft.repository.ast.TreeNodeRoot;
import org.deft.repository.ast.decoration.astname.AstNameInformation;
import org.eclipse.jdt.core.compiler.IProblem;
import org.eclipse.jdt.core.dom.ASTNode;
import org.eclipse.jdt.core.dom.ASTVisitor;
import org.eclipse.jdt.core.dom.Block;
import org.eclipse.jdt.core.dom.BodyDeclaration;
import org.eclipse.jdt.core.dom.CharacterLiteral;
import org.eclipse.jdt.core.dom.ChildListPropertyDescriptor;
import org.eclipse.jdt.core.dom.ChildPropertyDescriptor;
import org.eclipse.jdt.core.dom.Comment;
import org.eclipse.jdt.core.dom.CompilationUnit;
import org.eclipse.jdt.core.dom.Expression;
import org.eclipse.jdt.core.dom.MethodDeclaration;
import org.eclipse.jdt.core.dom.Modifier;
import org.eclipse.jdt.core.dom.NumberLiteral;
import org.eclipse.jdt.core.dom.PrimitiveType;
import org.eclipse.jdt.core.dom.SimpleName;
import org.eclipse.jdt.core.dom.SimplePropertyDescriptor;
import org.eclipse.jdt.core.dom.Statement;
import org.eclipse.jdt.core.dom.StringLiteral;
import org.eclipse.jdt.core.dom.StructuralPropertyDescriptor;
import org.eclipse.jdt.core.dom.TypeDeclaration;
import org.eclipse.jdt.core.dom.Annotation;

//TODO I: modified
public class AstConverterVisitor extends ASTVisitor {
	private static final String[][] tokens = new String[][] {
		{"(", "LPAREN"}, {")", "RPAREN"}, {"{", "LBRACE"}, {"}", "RBRACE"},
		{"[", "LBRACKET"}, {"]", "RBRACKET"}, {"@", "AT"}, {",", "COMMA"}, 
		{".", "DOT"}, {";", "SEMICOLON"}, {"...", "ELLIPSE"}, {"?", "QUESTION"}, {":", "COLON"},
		{"=", "ASSIGN"}, {"!", "EXCLAM"}, {"&", "AMPERSAND"}, {"|", "PIPE"}, 
		{"&&", "LOGICALAND"}, {"||", "LOGICALOR"}, 
		{"+", "PLUS"}, {"-", "MINUS"}, {"*", "MUL"}, {"/", "DIV"},
		{"++", "INCREMENT"}, {"--", "DECREMENT"}, {"+=", "ADDASSIGN"}, 
		{"-=", "SUBASSIGN"}, {"*=", "MULASSIGN"}, {"/=", "DIVASSIGN"}, 
		{"|=", "BITWISEORASSIGN"}, {"&=", "BITWISEANDASSIGN"},
		{"==", "EQUALS"}, {"!=", "NOTEQUALS"}, {"<", "LT"}, {">", "GT"}, 
		{"<=", "LE"}, {">=", "GE"}, {".*", "DOTSTAR"}, 
		{"<<", "LEFTSHIFT"}, {">>", "RIGHTSHIFT"}, {">>>", "SIGNEDRIGHTSHIFT"},
		{"<<=", "LEFTSHIFTASSIGN"}, {">>=", "RIGHTSHIFTASSIGN"}, 
		{">>>=", "SIGNEDRIGHTSHIFTASSIGN"},
		
		{"null", "NULL"}, {"super", "SUPER"}, {"this", "THIS"}, 
		{"true", "TRUE"}, {"false", "FALSE"},
		{"package", "PACKAGE"}, {"import", "IMPORT"}, {"static", "STATIC"},  
		{"class", "CLASS"}, {"interface", "INTERFACE"}, {"enum", "ENUM"}, 
		{"extends", "EXTENDS"}, {"implements", "IMPLEMENTS"},
		{"private", "PRIVATE"}, {"protected", "PROTECTED"}, {"public", "PUBLIC"},
		{"abstract", "ABSTRACT"}, {"final", "FINAL"}, {"const", "CONST"},
		{"byte", "BYTE"}, {"short", "SHORT"}, {"char", "CHAR"}, {"int", "INT"}, 
		{"long", "LONG"}, {"float", "FLOAT"}, {"double", "DOUBLE"}, 
		{"boolean", "BOOLEAN"}, {"void", "VOID"},
		{"new", "NEW"}, {"instanceof", "INSTANCEOF"}, 
		{"throw", "THROW"}, {"throws", "THROWS"}, {"try", "TRY"}, 
		{"catch", "CATCH"}, {"finally", "FINALLY"},
		{"if", "IF"}, {"else", "ELSE"}, {"for", "FOR"}, {"while", "WHILE"}, 
		{"do", "DO"}, {"goto", "GOTO"},
		{"switch", "SWITCH"}, {"case", "CASE"}, {"default", "DEFAULT"},
		{"break", "BREAK"}, {"continue", "CONTINUE"}, {"return", "RETURN"},
		{"assert", "ASSERT"}, {"synchronized", "SYNCHRONIZED"}, 
		{"transient", "TRANSIENT"}, {"strictfp", "STRICTFP"}, 
		{"volatile", "VOLATILE"}, {"native", "NATIVE"}	};
	
	private List<Comment> comments = new LinkedList<Comment>();
	private TreeNode firstNode ;
	private char[] input;
	private static final Set<String> modifiers;
	private Set<TreeNode> namedNodes = new HashSet<TreeNode>();
	
	static {
		modifiers = new HashSet<String>();
		modifiers.add("ABSTRACT"); 
		modifiers.add("FINAL");
		modifiers.add("NATIVE");
		modifiers.add("PRIVATE"); 
		modifiers.add("PROTECTED");
		modifiers.add("PUBLIC"); 
		modifiers.add("STRICTFP");
		modifiers.add("STATIC");
		modifiers.add("SYNCHRONIZED");
		modifiers.add("TRANSIENT");
		modifiers.add("VOLATILE");
	}
	
	private PositionConverter positionConverter;
	
	private Stack<TreeNodeContainer> stack = new Stack<TreeNodeContainer>();
	
	public AstConverterVisitor(char[] input) {
		this.input = input;
		BufferedReader reader = new BufferedReader(new CharArrayReader(input));
		try {
			positionConverter = new PositionConverter(reader);
		} catch (IOException e) {
			throw new RuntimeException("Error while trying to read input stream.", e);
		}
	}
	
	public AstConverterVisitor(char[] input, PositionConverter pConv) {
		this.input = input;
		this.positionConverter = pConv;
	}
	


	/**
	 * Converts an ASTNode that represents a token to a TokenNode. This method
	 * is necessary to convert multiline tokens (Multiline comments) appropriately.
	 * 
	 * This method is used for token nodes whose name does not have a direct correlation to
	 * its text. To be precise, this method is used for comments, string literals and number literals.
	 * 
	 * @param node
	 * @return
	 */
	private TokenNode convertTokenString(String name, ASTNode node) {
		String string = new String(input, node.getStartPosition(), node.getLength());
		Token token = new Token(string, convertPosition(node.getStartPosition(), getEndPosition(node))); 
		TokenNode result = new TokenNode(name, token);
		return result;
	}
	
	/**
	 * Converts a JDT AST node to a tree node. Most nodes are transformed 1:1, their
	 * class name used as the tree node's name. Some nodes, however, are renamed
	 * or otherwise specially treated (e.g. some token nodes) to receive a nicer
	 * tree node AST in the end.
	 * 
	 * @param node
	 * @return
	 */
	private TreeNode convertTreeNode(ASTNode node) {
		String name = node.getClass().getSimpleName();
		TreeNode child;
		if (node instanceof Modifier) {
			//a modifier node such as abstract or public
			//make the token node name ABSTRACT or PUBLIC or similar
			Modifier mod = (Modifier)node;
			String text = mod.getKeyword().toString();
			child = new TokenNode(text.toUpperCase(),
					createCheckedToken(text, node.getStartPosition()));
		} else if (node instanceof PrimitiveType) {
			//a primitive type such as int or boolean
			//make the token node name INT or BOOLEAN or similar
			PrimitiveType pt = (PrimitiveType)node;
			String text = pt.getPrimitiveTypeCode().toString();
			child = new TokenNode(text.toUpperCase(),
					createCheckedToken(text, node.getStartPosition()));
		} else if (node instanceof SimpleName) {
			//rename SimpleName to Identifier
			child = new TokenNode("Identifier",
					createCheckedToken(((SimpleName)node).getIdentifier(), node.getStartPosition()));
		} else if (node instanceof NumberLiteral || node instanceof StringLiteral || node instanceof CharacterLiteral) {
			//Make token nodes from number, string and character literals 
			child = convertTokenString(name, node);
		} else if (node instanceof TypeDeclaration) {
			//do not use TypeDeclaration, use either InterfaceDecl or ClassDecl
			TypeDeclaration td = (TypeDeclaration)node;
			if (td.isInterface()) {
				child = new TreeNode("InterfaceDeclaration");
			} else {
				child = new TreeNode("ClassDeclaration");
			}
		} else if (node instanceof MethodDeclaration) {
			MethodDeclaration md = (MethodDeclaration)node;
			if (md.isConstructor()) {
				child = new TreeNode("ConstructorDeclaration");
			} else {
				child = new TreeNode("MethodDeclaration");
			}
		} else if (node instanceof Block) {
			String parentName = node.getParent().getClass().getSimpleName();
			if (parentName.equals("MethodDeclaration")) {
				child = new TreeNode("Body");
			} else {
				child = new TreeNode(name);
			}
		} else {
			child = new TreeNode(name);
		}
		return child;
	}

	
	@SuppressWarnings("unchecked")
	private Map<Integer, String[]> findAllTokenPositions(ASTNode node) {		
		HashMap<Integer, String[]> result = new HashMap<Integer, String[]>();
		List<StructuralPropertyDescriptor> list = node.structuralPropertiesForType();
					
		//find children
		LinkedList<ASTNode> children = new LinkedList<ASTNode>();
		
		for (StructuralPropertyDescriptor desc : list) {
			
			if (desc instanceof SimplePropertyDescriptor)
				continue;
			
			Object object = node.getStructuralProperty(desc);
			if (object == null)
				continue;					
						
			if (desc instanceof ChildPropertyDescriptor)
				children.add((ASTNode) object);
			
			
			if (desc instanceof ChildListPropertyDescriptor)
				children.addAll((Collection<? extends ASTNode>) object);
		}
		
		//The following algorithm depends on the children to be sorted, so
		//make sure they are. While the children are usually sorted by default,
		//this is not always the case
		//E.g. in the DoStatement comes first the boolean condition
		//and then the statement/block, while in the code they appear the
		//other way round
		Collections.sort(children, new AstNodeComparator());
		
		int firstPosToCheck = node.getStartPosition();
		int lastPosToCheck = children.size() == 0 ? getEndPosition(node) + 1 : children.getFirst().getStartPosition();				
		addTokensInRange(firstPosToCheck, lastPosToCheck, result);
				
		//if we have no children we have to check the last position too.
		//otherwise this check is performed by searching after the last child
		if (children.size() == 0) 
			return result;
				
		//search in "holes" between children
		ASTNode last = children.getFirst();
		for (ASTNode child : children) {
			if (child == last)
				continue;

			firstPosToCheck = getEndPosition(last) + 1;
			lastPosToCheck = child.getStartPosition();						
			addTokensInRange(firstPosToCheck, lastPosToCheck, result);	

			last = child;
		}

		//search after last child
		firstPosToCheck = getEndPosition(last) + 1;
		lastPosToCheck = getEndPosition(node) + 1;					
		addTokensInRange(firstPosToCheck, lastPosToCheck, result);
		
		return result;
	}	

	/**
	 * Search for tokens within a given region in the input buffer.
	 * @param firstPosToCheck Defines the range to search.
	 * @param lastPosToCheck Defines the range to search.
	 * @param foundTokens Adds all tokens that have been found to this map.
	 */
	private void addTokensInRange(int firstPosToCheck, int lastPosToCheck, Map<Integer, String[]> foundTokens) {
		int windowStart = firstPosToCheck;
		while (windowStart >= firstPosToCheck) {
			
			while (symbolCanBeIgnoredByTokenAnalyser(windowStart))
				windowStart++;
			
			if (windowStart > lastPosToCheck)
				break;
			int highestEnd = -1;
			for (String[] tokenGroup : tokens) {
				String tokenText = tokenGroup[0]; 
				int windowEnd = windowStart + tokenText.length();
				if (windowEnd > lastPosToCheck)
					continue;
				int pos = findTokenPos(tokenText, windowStart, lastPosToCheck);
				//we found a token at position pos
				if (pos >= 0) {
					if (pos > highestEnd)
						highestEnd = windowEnd;
					foundTokens.put(pos, tokenGroup);					
				}
			}
			windowStart = highestEnd;
		}
	}
	
	/**
	 * Checks if a symbol at the given position can be ignored while
	 * searching the input buffer for occurring tokens.  
	 * @param pos
	 * @return true if the symbol can be ignored, false if the end of the buffer is reacher or
	 * the symbol can not be ignored.
	 */
	private boolean symbolCanBeIgnoredByTokenAnalyser(int pos) {
		if (pos >= input.length)
			return false;
		return input[pos] <= 32;
	}
	
	/**
	 * Returns the position of a token string in a given range of the input or -1
	 * if the token string could not be found.
	 * 
	 * If the token string could be found, but only as part of a bigger token, 
	 * it is not returned. E.g. if "int" is searched and found, but as part of "interface",
	 * only -1 is returned denoting a miss.
	 * 
	 * @param token the token string to be searched
	 * @param startPos start offset of search
	 * @param endPos end offset of search
	 * @return the position of the token string or -1 if it was not found
	 */
	private int findTokenPos(String token, int startPos, int endPos) {
		int result = findInput(token, startPos, endPos);
		
		if (result < 0)
			return -1;
		
		//token should not be in a comment
		if (isInComments(result))
			return -1;

		//the token should not be part of another token
		for (String[] tokenGroup2 : tokens) {
			String token2 = tokenGroup2[0];
			if (token2.length() <= token.length())
				continue;		
			if (!token2.contains(token))
				continue;
			int pos2 = findTokenPos(token2, startPos, endPos);
			if ((result >= pos2 && (result <= pos2 + token2.length())))
				return -1;
		}
		return result;
	}

	
	private int getEndPosition(ASTNode node) {
		return node.getStartPosition() + node.getLength() - 1; 
	}

	/**
	 * Returns whether a position (offset in the input file) is inside a comment
	 * 
	 * @param pos the position to be checked
	 * @return true if pos is inside a comment, otherwise false
	 */
	private boolean isInComments(int pos) {
		for (Comment c : comments)
			if ((c.getStartPosition() <= pos) && (pos <= getEndPosition(c)))
				return true;
		return false;
	}

	public final void checkInput(String expectedName, int pos) {
		for (int i = 0; i < expectedName.length(); i++)
			if (expectedName.charAt(i) != input[pos + i])
				throw new RuntimeException("Found character at position " + pos + " was not equal to the expected one.");
	}

	public Position convertPosition(int startPos, int endPos) {
		Position result = new Position(
				positionConverter.getLine(startPos),
				positionConverter.getCol(startPos),
				positionConverter.getLine(endPos), 
				positionConverter.getCol(endPos),
				startPos,
				endPos - startPos
				);
		 
		return result;
	}

	public Token createCheckedToken(String text, int pos) {
		checkInput(text, pos);
		return new Token(text, convertPosition(pos, pos + text.length()));
	}

	public TokenNode createCheckedTokenNode(String name, String text, int pos) {
		return new TokenNode(name, createCheckedToken(text, pos));
	}

	public int findInput(String expectedName, int start, int end) {
		if (end > input.length)
			return -1;
		
		if (end - start < expectedName.length())
			return -1;
		
		int count = 0;
		for (int i = start; i < end; i++) {
			if (count == expectedName.length())
				return i - count;
			
			if (expectedName.charAt(count) != input[i]) {
				i -= count;
				count = 0;
			}
			else
				count++;
		}
		
		if (count == expectedName.length())
			return end - count;
		
		return -1;
	}

	private boolean astNameInformationAdded = false;
	
	public TreeNode getTree() {
		if (!astNameInformationAdded) {
			//add tree node root to compilationUnit
			new TreeNodeRoot(firstNode);  //TODO
			for (TreeNode treeNode : namedNodes) {
				addAstNameInformation(treeNode);
			}
			astNameInformationAdded = true;
		}
		return firstNode;
	}

	@Override
	public void postVisit(ASTNode astnode) {
		TreeNodeContainer container = stack.pop();			
		
		for (TreeNode treeNode : container.getSortedChildren()) {
			TreeNode parent = container.getTreeNode();
			addChild(parent, treeNode);
			storeIfAstNameInformationCandidate(treeNode);
		}

		super.postVisit(astnode);
		
	}


	//TODO I: Kommentare??
	@Override
	public void preVisit(ASTNode node) {	
		
		if (node instanceof CompilationUnit) {
			for (Object c : ((CompilationUnit)node).getCommentList()) {
				comments.add((Comment) c);
			}
			firstNode= new TreeNode("CompilationUnit");
			stack.push(new TreeNodeContainer(firstNode));
			super.preVisit(node);
			return;
		}
		else if (node instanceof TypeDeclaration && stack.empty())
		{
			/*
			 * TODO P:1 Funktioniert nicht. s. auskommentierten Test javaparserhelper.testcase
			 */
			firstNode= new TreeNode("TypeDeclaration");
			stack.push(new TreeNodeContainer(firstNode));
			super.preVisit(node);
			return;
		}
		else if (node instanceof Block && stack.empty())
		{
			firstNode= new TreeNode("Block");
			stack.push(new TreeNodeContainer(firstNode));
			super.preVisit(node);
			return;
		}
		else if (node instanceof Expression && stack.empty())
		{
			firstNode= new TreeNode("Expression");
			stack.push(new TreeNodeContainer(firstNode));
			super.preVisit(node);
			return;
		}
		
		
		TreeNodeContainer parent = stack.peek();

		TreeNode child = convertTreeNode(node);
		parent.addChild(node.getStartPosition(), child);
				
		
		stack.push(new TreeNodeContainer(child));
		if (child instanceof TokenNode)
			return;
		
		addAllTokens(node);
		super.preVisit(node);
	}
	
	
	
	private void addAllTokens(ASTNode node) {
		TreeNodeContainer container = stack.peek();
		Map<Integer, String[]> positions = findAllTokenPositions(node);
		for (Integer pos : positions.keySet()) {
			String[] s = positions.get(pos);
			String text = s[0];
			String name = s[1];
			container.addChild(pos, createCheckedTokenNode(name, text, pos));
		}
	}

	@Override
	public boolean visit(CompilationUnit node) {
		IProblem[] problems = node.getProblems();
		for (int i = 0; i < problems.length; i++) {
			System.err.println("Line[" + problems[i].getSourceLineNumber() + "] :" + problems[i].getMessage());
			System.out.println(problems[i].getClass());
		}
		return super.visit(node);
		
	}


	
	
	/**
	 * Returns a (degenerated) tree containing the comments. All encountered comments
	 * are children of the node CompilationUnit.
	 * 
	 * @return a tree with comments
	 */
	public TreeNode getCommentTree() {
		TreeNode tnCu = new TreeNode("CompilationUnit");
		for (Comment c : comments) {
			String name = c.getClass().getSimpleName();
			TreeNode tn = null;
			if (name.equals("Javadoc")) {
				tn = convertTokenString("JAVADOC", c);
			} else {
				tn = convertTokenString("COMMENT", c);
			}
			tnCu.addChild(tn);
		}
		return tnCu;
	}
	
	/**
	 * Adds a tree node to a parent tree node. For some nodes a minor tree rewrite is 
	 * desired. This is done here. For example, modifiers such as PUBLIC are grouped
	 * under a Modifier node.
	 * 
	 * @param parent
	 * @param child
	 */
	private void addChild(TreeNode parent, TreeNode child) {
		if (modifiers.contains(child.getName())) {
			introduceGroupingNode(parent, child, "Modifiers");
		} else if (child.getName().equals("ImportDeclaration")){
			introduceGroupingNode(parent, child, "ImportDeclarations");
		} else if (parent.getName().equals("IfStatement")){
			if (parent.getChildCount() == 4) {
				TreeNode tnIfSection = new TreeNode("IfSection");
				parent.addChild(tnIfSection);
				tnIfSection.addChild(child);
			} else if (parent.getChildCount() == 6) {
				TreeNode tnElseSection = new TreeNode("ElseSection");
				parent.addChild(tnElseSection);
				tnElseSection.addChild(child);				
			} else {
				parent.addChild(child);
			} 
		} else if (parent.getName().equals("SwitchStatement")) {
			//check if we have already added: switch ( id ) {
			if (parent.getChildCount() < 5) {
				parent.addChild(child);
			} else if (child.getName().equals("RBRACE")) { //closing brace?
				parent.addChild(child);
			} else {
				//we have switch cases or statements to be added to SwitchSections
				//we have a switch case
				if (child.getName().equals("SwitchCase")) {
					TreeNode tnLastChild = parent.getLastChild();
					//we already have at least one SwitchSection
					if (tnLastChild.getName().equals("SwitchSection")) {
						TreeNode tnLastInSection = tnLastChild.getLastChild();
						//if last child in section was a switch case, we add
						//this one to the same section
						if (tnLastInSection.getName().equals("SwitchCase")) {
							tnLastChild.addChild(child);
						} else {
							//last child in section was some statement,
							//so make new section and add the switch case to it
							TreeNode tnNewSwitchSection = new TreeNode("SwitchSection");
							parent.addChild(tnNewSwitchSection);
							tnNewSwitchSection.addChild(child);
						}						
					} else {
						//no switch section yet, create first one
						tnLastChild = new TreeNode("SwitchSection");
						parent.addChild(tnLastChild);
						tnLastChild.addChild(child);						
					}
				} else {
					//we have sth from the switch body, add it to the last
					//existing switch section
					TreeNode tnSwitchSection = parent.getLastChild();
					assert tnSwitchSection.getName().equals("SwitchSection") 
							: "Expected SwitchSection, but AST contains " 
							+ tnSwitchSection.getName();
					tnSwitchSection.addChild(child);
				}
			}
		} else if (parent.getName().equals("TryStatement")) {
			//TRY token and the block after the try token
			TreeNode tLastChild = parent.getLastChild();
			String childName = child.getName();
			//check whether to add to TryBlock
			if (childName.equals("TRY") 
					|| (tLastChild != null && tLastChild.getName().equals("TryBlock"))
					&& !childName.equals("FINALLY") 
					&& !childName.equals("CatchClause")) {
				introduceGroupingNode(parent, child, "TryBlock");
			}
			//check whether to add to FinallyClause
			else if (childName.equals("FINALLY") 
					|| ((tLastChild != null && tLastChild.getName().equals("FinallyClause"))
							&& !childName.equals("TRY") 
							&& !childName.equals("CatchClause")	
					)) {
				introduceGroupingNode(parent, child, "FinallyClause");
			} else {
				//otherwise we have a CatchClause. This one is already properly grouped,
				//so add it directly
				parent.addChild(child);
			}
		} else if (isAnnotation(child)) {
			introduceGroupingNode(parent, child, "Annotations");
		} else if (isTypeParametersClause(parent, child)) {
			introduceGroupingNode(parent, child, "TypeParameters");
		} else if (isFormalParamtersClause(parent, child)) {
			introduceGroupingNode(parent, child, "Parameters");
		} else if (isThrowsClause(parent, child)) {
			introduceGroupingNode(parent, child, "ThrowsClause");
		} else if (isExtendsClause(parent, child)) {
			introduceGroupingNode(parent, child, "Extends");
		} else if (isImplementsClause(parent, child)) {
			introduceGroupingNode(parent, child, "Implements");
		} else if (isTypeDeclaration(parent) || parent.getName().equals("EnumDeclaration")
				|| parent.getName().equals("AnnotationTypeDeclaration")) {
			TreeNode lastChild = parent.getLastChild();
			if (child.getName().equals("LBRACE") 
					|| (lastChild != null && lastChild.getName().equals("Body"))) {
				introduceGroupingNode(parent, child, "Body");
			} else {
				parent.addChild(child);
			}
		} else {
			parent.addChild(child);
		}
	}
	

	
	private boolean isExtendsClause(TreeNode parent, TreeNode child) {
		String parentName = parent.getName();
		String childName = child.getName();
		if (parentName.equals("ClassDeclaration") || parentName.equals("InterfaceDeclaration")) {
			if (childName.equals("EXTENDS")) {
				return true;
			} else {
				TreeNode lastChild = parent.getLastChild();
				if (lastChild != null && lastChild.getName().equals("Extends") && !childName.equals("IMPLEMENTS") 
						&& !childName.equals("LBRACE") && !child.getName().equals(";")) {
					return true;
				} else {
					return false;
				}
			}
		} else {
			return false;
		}
	}
	
	private boolean isImplementsClause(TreeNode parent, TreeNode child) {
		String parentName = parent.getName();
		String childName = child.getName();
		if (parentName.equals("ClassDeclaration") || parentName.equals("InterfaceDeclaration")) {
			if (childName.equals("IMPLEMENTS")) {
				return true;
			} else {
				TreeNode lastChild = parent.getLastChild();
				if (lastChild != null && lastChild.getName().equals("Implements") && 
						!childName.equals("LBRACE") && !child.getName().equals(";")) {
					return true;
				} else {
					return false;
				}
			}
		} else {
			return false;
		}
	}
	
	private boolean isThrowsClause(TreeNode parent, TreeNode child) {
		String parentName = parent.getName();
		String childName = child.getName();
		if (parentName.equals("MethodDeclaration") || parentName.equals("ConstructorDeclaration")) {
			//if we find "throws" and do not yet have a body or a ;, we are in the throws clause
			if (childName.equals("THROWS")) {
				return true;
			} else {
				TreeNode lastChild = parent.getLastChild();
				if (lastChild != null && lastChild.getName().equals("ThrowsClause") && 
						!childName.equals("Body") && !childName.equals(";")) {
					return true;
				} else {
					return false;
				}
			}
		} else {
			return false;
		}
	}
	
	private boolean isFormalParamtersClause(TreeNode parent, TreeNode child) {
		String parentName = parent.getName();
		String childName = child.getName();
		if (parentName.equals("MethodDeclaration") || parentName.equals("ConstructorDeclaration")) {
			//if we find ( or ) we are definitely in the formal parameters clause
			if (childName.equals("LPAREN") || childName.equals("RPAREN")) {
				return true;
			} else {
				TreeNode lastChild = parent.getLastChild();
				if (lastChild != null && lastChild.getName().equals("Parameters") &&
						!lastChild.getLastChild().getName().equals("RPAREN")) {
					return true;
				} else {
					return false;
				}
			}
		} else {
			return false;
		}
	}
	
	private boolean isTypeParametersClause(TreeNode parent, TreeNode child) {
		String parentName = parent.getName();
		String childName = child.getName();
		if (parentName.equals("MethodDeclaration") || parentName.equals("ConstructorDeclaration") ||
				parentName.equals("ClassDeclaration") || parentName.equals("InterfaceDeclaration")) {
			//if we find < or > we are definitely in the TypeParameters clause
			if (childName.equals("LT") || childName.equals("GT")) {
				return true;
			} else {
				TreeNode lastChild = parent.getLastChild();
				if (lastChild != null && lastChild.getName().equals("TypeParameters") &&
						!lastChild.getLastChild().getName().equals("GT")) {
					return true;
				} else {
					return false;
				}
			}
		} else {
			return false;
		}
	}
	
	private boolean isAnnotation(TreeNode node) {
		String name = node.getName();
		return name.equals("SingleMemberAnnotation") || name.equals("MarkerAnnotation") ||
				name.equals("NormalAnnotation");
	}
	
	private boolean isTypeDeclaration(TreeNode node) {
		String name = node.getName();
		return name.equals("ClassDeclaration") || name.equals("InterfaceDeclaration");
	}
	
	private boolean isMethodDeclaration(TreeNode node) {
		String name = node.getName();
		return name.equals("MethodDeclaration") || name.equals("ConstructorDeclaration");
	}
	


	/**
	 * Adds a node as a child to a grouping node, which in turn is a child of parent node.
	 * If parent already has a grouping node (because this method has previously been
	 * called and the grouping node added there), this grouping node is used to add child to.
	 * 
	 * @param parent parent node under which the grouping node should be added
	 * @param child child node, to be added to the grouping node
	 * @param groupingNodeName name of the grouping node to be created or reused
	 */
	private void introduceGroupingNode(TreeNode parent, TreeNode child, String groupingNodeName) {
		TreeNode groupingNode = null;
		//find grouping node as child of parent node
		for (TreeNode tn : parent.getChildren()) {
			if (tn.getName().equals(groupingNodeName)) {
				groupingNode = tn;
				break;
			}
		}
		//if there was no grouping node, add it. 
		if (groupingNode == null) {
			groupingNode = new TreeNode(groupingNodeName);
			parent.addChild(groupingNode);
		}
		//finally, add the child node to the grouping node
		groupingNode.addChild(child);
	}
	
	private void storeIfAstNameInformationCandidate(TreeNode node) {
		String name = node.getName();
		if (isTypeDeclaration(node)
				|| name.equals("EnumDeclaration") || name.equals("AnnotationTypeDeclaration")
				|| isMethodDeclaration(node)
				|| name.equals("FieldDeclaration") || name.equals("AnnotationTypeMemberDeclaration")) {
			namedNodes.add(node);
		}
	}
	
	private void addAstNameInformation(TreeNode node) {
		String name = node.getName();
		if (name.equals("MethodDeclaration") || name.equals("ConstructorDeclaration")) {
			addMethodAstNameInformation(node);
		} else if (name.equals("FieldDeclaration")) {
			addFieldAstNameInformation(node);
		} else {
			addSimpleAstNameInformation(node);
		}
	}
	
	private void addSimpleAstNameInformation(TreeNode node) {
		TokenNode ident = (TokenNode)node.executeXPathQuery("Identifier").get(0);
		String s = ident.getTextContent();
		s = s.replaceAll("\\s", ""); //remove all whitespace
		AstNameInformation ani = new AstNameInformation(s);
		node.addInformation(ani);
	}
	
	private void addMethodAstNameInformation(TreeNode node) {
		TokenNode ident = (TokenNode)node.executeXPathQuery("Identifier").get(0);
		//get the second-to-last children of all SingleVariableDeclarations
		//(usually there are only 2 children and the type that we need for the signature
		//is the first child. However, sometimes there is a final parameter which
		//screws things up. Therefore go for the second-to-last children
		List<TreeNode> types = node.executeXPathQuery("Parameters/SingleVariableDeclaration");
		StringBuilder sb = new StringBuilder();
		sb.append(ident.getTextContent());
		sb.append("(");
		for (int i = 0; i < types.size(); i++) {
			if (i > 0) {
				sb.append(",");
			}
			TreeNode tn = types.get(i);
			TreeNode tnType = tn.getFirstChild();
			if (tnType.getName().equals("Modifiers")) {
				tnType = tn.getChild(1);
			}
			sb.append(tnType.getTextContent());
			List<TreeNode> tnEllipse = tn.executeXPathQuery("ELLIPSE");
			if (!tnEllipse.isEmpty()) {
				sb.append("...");
			}
		}
		sb.append(")");
		String signature = sb.toString().replaceAll("\\s", ""); //remove all whitespace;
		AstNameInformation ani = new AstNameInformation(signature);
		node.addInformation(ani);
	}
	
	private void addFieldAstNameInformation(TreeNode node) {
		List<TreeNode> list = node.executeXPathQuery(
				"VariableDeclarationFragment/Identifier");
		StringBuilder sb = new StringBuilder();
		for (int i = 0; i < list.size(); i++) {
			TreeNode tn = (TokenNode)list.get(i); //is actually a token node
			if (i > 0) {
				sb.append(",");
			}
			sb.append(tn.getTextContent());
		}
		AstNameInformation ani = new AstNameInformation(sb.toString());
		node.addInformation(ani);
	}
	
	private class AstNodeComparator implements Comparator<ASTNode> {

		@Override
		public int compare(ASTNode n1, ASTNode n2) {
			int n1Offset = n1.getStartPosition();
			int n2Offset = n2.getStartPosition();
			if (n1Offset < n2Offset) {
				return -1;
			}
			if (n1Offset > n2Offset) {
				return 1;
			}
			return 0;
		}
		
	}
	

}
