package org.deft.language.java.parser;

import java.io.BufferedReader;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.util.LinkedList;
import java.util.List;

public class PositionConverter {
    
    private List<Integer> lineStarts;
    
    
    public PositionConverter(BufferedReader sourceCodeReader) throws IOException {
        makeLineStarts(sourceCodeReader);
        sourceCodeReader.close();
    }
    
    private void makeLineStarts(BufferedReader sourceCodeReader) 
            throws FileNotFoundException, IOException {
        lineStarts = new LinkedList<Integer>();
        lineStarts.add(0);
        
        int i = 1;
        while (sourceCodeReader.ready()) {
            //this extendedReadLine does not discard the linebreaks;
            //this is important because linebreaks can be either 1 character long 
            //(\n in unix, \r in mac) or 2 characters long (\r\n in win), 
            //screwing up the carefully computed position information
            String s = extendedReadLine(sourceCodeReader);
            lineStarts.add(lineStarts.get(lineStarts.size()-1) + s.length());
            i++;
        }
 
    }
    
    
    private String extendedReadLine(BufferedReader br) {
        int read = -1;
        StringBuilder sb = new StringBuilder();
        try {
            //while there is still sth to read and we have not yet reached end of line 
            while (br.ready() && ((char)read != '\r') && ((char)read != '\n')) {
                //read and append to return string
                read = br.read();
                //0xfeff is a special character that is used in UTF-16 to mark the
                //byte order (Big Endian). It is a zero width character, so it cannot be
                //seen. But if it is not ignored here, the line number computations are
                //screwed up.
                if (read != 0xfeff) {
                    sb.append((char)read);
                }
            }
            //if we are here, an eol char has been found, either \r or \n;
            //if it was \r, check whether the next char is \n
            //(because the combination \r\n is also a valid eol sequence)
            if ((char)read == '\r' && br.ready()) {
                //remember position in stream
                br.mark(1);
                char nextRead = (char)br.read();
                //we found a \n, append to string
                if (nextRead == '\n') {
                    sb.append(nextRead);
                } else {
                    //no \n found, the character read was the first char from
                    //the next line ... so set stream back before that char
                    br.reset();
                }
            }
        }
        catch (IOException ioe) {
            ioe.printStackTrace();
        }
        return sb.toString();
    }
    
    
    public int getLine(int position) {
        for (int i = 0; i < lineStarts.size(); i++) {
            if (lineStarts.get(i) > position) {
                return i;
            }
        }
        return -1;
    }
    
    public int getCol(int position) {
        int line = getLine(position);
        if (line != -1) {
            return position - lineStarts.get(line-1) + 1;
        } else {
            return -1;
        }
    }
    
    public int getOffset(int line, int col) {
        int offset = -1;
        if (line < lineStarts.size()) {
            int lineOffset = lineStarts.get(line);
            offset = lineOffset + col;
        }
        return offset;
    }   

}
