package org.deft.repository.ast;

import java.util.ArrayList;
import java.util.Collections;
import java.util.LinkedList;
import java.util.List;

import org.deft.repository.ast.decoration.NodeInformation;

public class AstCombiner {
    
	
    private List<TokenNode> serializedNodeList;
    private MergeConfiguration config;
    
    public void combineTrees(TreeNode baseTree, TreeNode foreignTree, MergeConfiguration config) {
        this.config = config;
        //make a list of Tokens from the baseTree, it is used to compute
        //the "extended offset"
        //Using an ArrayList instead of the standard LinkedList offers a speedup
        //of about factor 100 (culprits are method getExtendedOffset and getExtendedEndOffset)
        serializedNodeList = new ArrayList<TokenNode>(baseTree.serialize());
        //lock trees, otherwise the nodes' offset information would be updated if
        //child nodes were removed or addedd
        baseTree.lock();
        foreignTree.lock();
        //the foreign tree might have introduced "virtual" token nodes (placeholder token nodes)
        //and "hooks" (placeholder tree nodes, with a virtual token node as child)
        //remove them
        removeVirtualTokens(baseTree);       //an empty baseTree contains virtual token node 
											 //(should happen almost never, though)
        removeVirtualTokens(foreignTree);    //an empty foreignTree contains virtual token node
        removeUnwantedNodes(foreignTree);
        //we want to iterate over the foreign tree's children and also remove them
        //(by adding them to the base tree); 
        //make a copy of the children list to avoid ConcurrentModificationException
        List<TreeNode> lForeignTreeNodes = copy(foreignTree.getChildren());
        for (TreeNode tn : lForeignTreeNodes) {
            foreignTree.removeChild(tn);
            //find the node in the baseTree that will be the new parent node for tn
            TreeNode nParentInBaseTreeForForeignNodes = findSpanningNode(baseTree, tn);
            insert(nParentInBaseTreeForForeignNodes, tn);
        }
        
        //if the removal of the foreign hooks left empty nodes, remove them.
        removeEmptyNodes(baseTree);
        //unlock base tree to have offsets computed on the fly again
        //(merging with the foreign tree might have introduced minor "conflicts")
        baseTree.unlock();
    }
    
    private void insert(TreeNode parent, TreeNode childToAdd) {

        //find the node under parent which is incompatible with childToAdd (might be null
        //if the nodes are compatible (standard case))
        TreeNode incompatibleSibling = getIncompatibleSibling(parent, childToAdd);
        //if nodes are compatible, just insert childToAdd at the right position
        if (incompatibleSibling == null) {
            //get all child nodes from parent, that have to be moved under childToAdd
            //(i.e. all of parent's child nodes that fall into childToAdd's offset range)
            List<TreeNode> lNodesToMove = findNodesToMove(parent, childToAdd);

            //find out at which position under parent the node childToAdd will be added
            int indexForChild = findIndex(parent, childToAdd);            
            parent.addChild(indexForChild, childToAdd);

            for (TreeNode tn : lNodesToMove) {
                parent.removeChild(tn);
                TreeNode newParent = findSpanningNode(childToAdd, tn);
                insert(newParent, tn);
            }
        } else {
            //nodes are incompatible, add debug message
//            logger.debug(incompatibleSibling + " (" + incompatibleSibling.getOffset() + ".." 
//                    + incompatibleSibling.getEndOffset() + ") and " + childToAdd + " (" + childToAdd.getOffset() 
//                    + ".." + childToAdd.getEndOffset() + ") are incompatible.");

            //incompatibleSibling under parent starts before childToAdd
            if (incompatibleSibling.getOffset() < childToAdd.getOffset()) {
                //split childToAdd, incompatibleSibling stays intact
                List<SplitTreeNode> lsplit = split(incompatibleSibling, childToAdd);
                SplitTreeNode split1 = lsplit.get(0);
                SplitTreeNode split2 = lsplit.get(1);
                //the split parts of childToAdd are now compatible with parent
                assert getIncompatibleSibling(incompatibleSibling, split1) == null;
                assert getIncompatibleSibling(incompatibleSibling, split2) == null;
                //Add split1 by recursively calling this method. But first find its new parent node
                //(split1 cannot be directly added to parent because the respective
                //space (offsets) is occupied by incompatibleSibling. Therefore find a node under
                //incompatibleSibling to which split can be added)
                split1.releaseFromParent();
                TreeNode newParent = findSpanningNode(incompatibleSibling, split1);
                insert(newParent, split1);                
                //add split2 at the right position under parent
                //(originally we wanted to add child to parent. As this
                //did not work out, add at least split2 to parent, which will, after the 
                //split from above, definitely be compatible and fit)
                split2.releaseFromParent();
                insert(parent, split2);
            } else {//child starts before incompatibleSibling
                //split incompatibleSibling, childToAdd stays intact
                List<SplitTreeNode> lSplit = split(childToAdd, incompatibleSibling);
                //split1 is the part of the split node that has to be relocated,
                //the other SplitTreeNode in lSplit is already at the correct position
                SplitTreeNode split1 = lSplit.get(0);
                
                //recursively call this method: add split1 at the correct position in childToAdd
                //But first find its new parent node.
                split1.releaseFromParent();
                TreeNode newParent = findSpanningNode(childToAdd, split1);
                insert(newParent, split1);

                //add child to parent, after all, this was our goal from the very beginning
                //(the arguments are the same as in the original method call, but this time
                //the two nodes are not incompatible anymore and child can safely be inserted
                //under parent
                insert(parent, childToAdd);
            }
        }
    }
 
    
    
    private List<TreeNode> findNodesToMove(TreeNode nMain, TreeNode nOther) {
        List<TreeNode> lRet = new LinkedList<TreeNode>();
        for (TreeNode n : nMain.getChildren()) {
            if (n.getOffset() >= nOther.getOffset()&& n.getEndOffset() <= nOther.getEndOffset()) {
                lRet.add(n);
            }
        }
        return lRet;
    }
    


    private TreeNode findSpanningNode(TreeNode root, TreeNode foreignNode) {
        int fitFrom = foreignNode.getOffset();
        int fitTo = foreignNode.getEndOffset();
        for (TreeNode tn : root.getChildren()) {
            int tnStart = determineStartOffset(tn);
            int tnEnd = determineEndOffset(tn);
            //System.out.println(tn + " " + tnStart + "   " + tnEnd);
            if (tnStart <= fitFrom && 
                    (tnEnd >= fitTo || (tnEnd == -1 && !isVirtual(tn)))) {
                return findSpanningNode(tn, foreignNode);
            }
            if (tnStart <= fitFrom && tnEnd <= fitTo && fitFrom <= tnEnd) {
                //System.out.println(root + " " + foreignNode + " " + tn);
            }

        }
        return root;
    }
    
    private List<SplitTreeNode> split(TreeNode nReference, TreeNode nodeToSplit) {
        assert nReference.getOffset() < nodeToSplit.getOffset() 
                && nReference.getEndOffset() > nodeToSplit.getOffset()
                : "Nodes nReference (" + nReference + ") and nToSplit (" + nodeToSplit 
                + ") are not incompatible, so nToSplit cannot be split";
        //split right after the reference node
        int splitOffset = nReference.getEndOffset();
        List<TreeNode> lNodesToSplit = findDescendantsToSplit(splitOffset, nodeToSplit);
        Collections.reverse(lNodesToSplit);
        for (TreeNode n : lNodesToSplit) {
            splitAt(splitOffset, n);
        }
        List<SplitTreeNode> list = splitAt(splitOffset, nodeToSplit);
        return list;    
    }
    
    

    

    private List<SplitTreeNode> splitAt(int splitOffset, TreeNode nodeToSplit) {
        List<SplitTreeNode> list = new LinkedList<SplitTreeNode>();
        SplitTreeNode n1 = new SplitTreeNode(nodeToSplit);
        SplitTreeNode n2 = new SplitTreeNode(nodeToSplit);
        if (nodeToSplit instanceof SplitTreeNode) {
            SplitTreeNode stn = (SplitTreeNode)nodeToSplit;
            SplitTreeNode oldSuccessor = stn.getSuccessor();

            n1.setSuccessor(n2);
            n2.setPredecessor(n1);            
            oldSuccessor.setPredecessor(n2);
            n2.setSuccessor(oldSuccessor);
        } else {
            n1.setSuccessor(n2);
            n2.setPredecessor(n1);
        }

        for (NodeInformation ni : nodeToSplit.getInformation()) {
            n1.addInformation(ni);
            n2.addInformation(ni);
        }
        if (nodeToSplit.hasChildren()) {
            for (TreeNode c : copy(nodeToSplit.getChildren())) {
                nodeToSplit.removeChild(c);
                if (c.getEndOffset() <= splitOffset) {
                    n1.addChild(c);
                } else {
                    n2.addChild(c);
                }
            }
            n1.lock();
            n2.lock();
        } else {
            n1.setEndOffset(splitOffset);
            n2.setOffset(splitOffset);
        }

        TreeNode parent = nodeToSplit.getParent();
        if (parent != null) {
            int pos = nodeToSplit.getSiblingIndex();
            parent.removeChild(nodeToSplit);
            parent.addChild(pos, n1);
            parent.addChild(pos + 1, n2);
        }
        list.add(n1);
        list.add(n2);
        return list;
    }

    

    private List<TreeNode> findDescendantsToSplit(int offset, TreeNode node) {
        List<TreeNode> list = new LinkedList<TreeNode>();
        for (TreeNode c : node.getChildren()) {
            if (c.getOffset() < offset && c.getEndOffset() > offset) {
                list.add(c);
                List<TreeNode> recursionResult = findDescendantsToSplit(offset, c);
                list.addAll(recursionResult);
                return list;
            }
        }
        return list;
    }
    
    
    

    
    private TreeNode getIncompatibleSibling(TreeNode parent, TreeNode child) {

        int cStart = child.getOffset();
        int cEnd = child.getEndOffset();

        for (int index = 0; index < parent.getChildCount(); index++) {               
            //check first kind of overlap: s starts, c starts, s ends, c ends
            TreeNode sibling = parent.getChild(index);
            int sStart = sibling.getOffset();
            int sEnd = sibling.getEndOffset();
            //check if child and preceding node (sibling) are incompatible
            if (sStart < cStart && sEnd > cStart && sEnd < cEnd) {
                return sibling;
            }
            
            //check second kind of overlap: c starts, s starts, c ends, s ends
            sStart = sibling.getOffset();
            sEnd = sibling.getEndOffset();
            //check if child and subsequent node (sibling) are incompatible
            if (cStart < sStart && cEnd > sStart && cEnd < sEnd) {
                return sibling;
            }
        }
        return null;
    }

    private List<TreeNode> getIncompatibleDescendants(TreeNode parent, TreeNode child) {
        List<TreeNode> list = findDescendantsToSplit(child.getOffset(), child);
        return list;
    }
    

    private int findIndex(TreeNode nMain, TreeNode nOther) {
        int startOffset = nOther.getOffset();
        for (int i = 0; i < nMain.getChildCount(); i++) {
            TreeNode child = nMain.getChild(i);
            if (child.getOffset() > startOffset) {
                return i;
            }
        }
        //nothing found, index is behind all other TreeNodes
        return nMain.getChildCount();
    }


    private int determineStartOffset(TreeNode node) {
        int offset = node.getOffset();
        //if this node is not configured for extended offset and the
        //extended offset is not forced, return the normal offset
        if (!config.isInsertBeforeForbidden(node.getName())) {
            return offset;
        }
        //otherwise compute the extended offset
        for (int i = 0; i < serializedNodeList.size(); i++) {
            TokenNode tn = serializedNodeList.get(i);
            int tnEndOffset = tn.getEndOffset();
            //loop until we find a TokenNode ending AFTER
            //the examined TreeNode starts
            if (tnEndOffset > offset) {
                if (i == 0) {
                	//already the first token was a match
                	//the extended start offset is the beginning
                	//of the file, i.e. 0
                    return 0;
                } else {
                	//found something
                	//the extended start offset is the end offset of the
                	//previous token
                    return serializedNodeList.get(i - 1).getEndOffset();
                }
            }
        }
        assert false;
        return -2;
    }
    
    private int determineEndOffset(TreeNode node) {
        int endOffset = node.getEndOffset();
        if (!config.isInsertAfterForbidden(node.getName())) {
            return endOffset;
        }
        for (int i = 0; i < serializedNodeList.size(); i++) {
            TokenNode tn = serializedNodeList.get(i);
            int tnOffset = tn.getOffset();
            //important: >=, not only >
            if (tnOffset >= endOffset) { 
                return tnOffset;
            }
            if (i == serializedNodeList.size() - 1) {
                return -1;  //we don't know how long the file is and how many "invisible" tokens
                            //(e.g. comments) are yet to come. Therefore indicate "infinite" with -1
            }
        }
        assert false;
        return -2;
    }    
    

    private boolean isVirtual(TreeNode node) {
        return node.getOffset() == -1 && node.getEndOffset() == -1;
    }
    
    private void removeVirtualTokens(TreeNode root) {
        for (TokenNode node : root.serialize()) {
            if (isVirtual(node)) {
                node.releaseFromParent();
            }
        }
    }
    
    /**
     * Removes all nodes whose name equals that of a foreign hook (as specified by the
     * MergeConfiguration). This is used to get rid of tree nodes from the foreign tree
     * that had to be included during the original tree creation but are now useless.
     * 
     * Usually the foreign hooks are the nodes that had virtual token nodes attached.
     * 
     * @param node
     */
    private void removeUnwantedNodes(TreeNode node) {
        //if currently examined node is a foreign hook node
        if (config.shallBeRemoved(node.getName()) && node.getParent() != null) {
            int pos = node.getSiblingIndex();
            TreeNode parent = node.getParent();
            parent.removeChild(node);
            int i = 0;
            //we want to cut out this node, so add all its children to its parent
            //also apply this method recursively to all children. Usually, however,
            //there are no children, because foreign hooks are leaf nodes (the virtual
            //nodes have already been removed).
            //But with the recursive call this method can be used
            //as a simple filter to remove nodes with special names.
            for (TreeNode nodeToMove : copy(node.getChildren())) {//make copy of to avoid ConcurrentModification
                parent.addChild(pos + i, nodeToMove);
                removeUnwantedNodes(nodeToMove);
                i++;
            }
        }
        //make copy avoid ConcurrentModificationException
        for (TreeNode n : copy(node.getChildren())) {
            removeUnwantedNodes(n);
        }       
    }
    
    private void removeEmptyNodes(TreeNode tree) {
        List<TreeNode> lNodes = new LinkedList<TreeNode>();
        for (TreeNode node : tree) {
            lNodes.add(node);
        }
        for (TreeNode node : copy(lNodes)) {
            while (!node.hasChildren() && !(node instanceof TokenNode) && node.getParent() != null) {
                TreeNode parent = node.getParent();
                parent.removeChild(node);
                node = parent;
            }
        }
    }
    
    private List<TreeNode> copy(List<TreeNode> list) {
        return new LinkedList<TreeNode>(list);
    }
    

}
