package org.deft.repository.ast.serialize;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.Stack;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.deft.repository.ast.SplitTreeNode;
import org.deft.repository.ast.Token;
import org.deft.repository.ast.TokenNode;
import org.deft.repository.ast.TreeNode;
import org.deft.repository.ast.TreeNodeRoot;
import org.deft.repository.ast.decoration.TemplatesBasic;
import org.deft.repository.ast.decoration.astname.AstNameInformation;



public class ParenTreeNodeSerializer implements ITreeNodeSerializer {

	
	/**
	 * Helper map for conversion of SplitTreeNodes to string representation
	 */
	private Map<SplitTreeNode, Integer> stnIdMap = new HashMap<SplitTreeNode, Integer>();
	
	/**
	 * Helper map for conversion of string representation to SplitTreeNodes
	 */
	private Map<SplitTreeNode, Integer> needsSuccessorMap = new HashMap<SplitTreeNode, Integer>();
	private Map<SplitTreeNode, Integer> needsPredecessorMap = new HashMap<SplitTreeNode, Integer>();
	private Map<Integer, SplitTreeNode> idStnMap = new HashMap<Integer, SplitTreeNode>();
	
	
	public TreeNodeRoot read(InputStream is) {
		InputStream iStream = is;
		if (!(is instanceof BufferedInputStream)) {
			iStream = new BufferedInputStream(is);
		}
		try {
			String s = getString(iStream);
			TreeNodeRoot tree = convertStringToTree(s);
			return tree;
		} catch (IOException e) {
			e.printStackTrace();
			return null;
		}
	}
	
	private String getString(InputStream in) throws IOException {
	    StringBuilder sb = new StringBuilder();
	    byte[] b = new byte[4096];
	    for (int n; (n = in.read(b)) != -1;) {
	        sb.append(new String(b, 0, n));
	    }
	    return sb.toString();
	}
	

	public void write(TreeNodeRoot root, OutputStream os) {
		String s = convertTreeToString(root);
		OutputStream oStream = os;
		boolean isBuffered = (os instanceof BufferedOutputStream); 
		if (!isBuffered) {
			oStream = new BufferedOutputStream(os);
		}
		try {
			oStream.write(s.getBytes());
			if (!isBuffered) {
				oStream.close();
			}
		} catch (IOException ioe) {
			ioe.printStackTrace();
		}	
	}
	
	public String convertTreeToString(TreeNode root) {
	    stnIdMap.clear();
		if (root instanceof TreeNodeRoot) {
			return doConvertTreeToString(root.getFirstChild());
		}
		String ret = doConvertTreeToString(root);
		stnIdMap.clear();
		return ret;
	}
	
	private String doConvertTreeToString(TreeNode root) {
        StringBuilder sb = new StringBuilder(); 
        sb.append(convertSingleNodeToString(root));
        if (root.hasChildren()) {
            sb.append("(");
            for (TreeNode child : root.getChildren()) {
                sb.append(doConvertTreeToString(child));
                sb.append(" ");
            }
            sb.deleteCharAt(sb.length()-1);
            sb.append(")");
        }
        return sb.toString();	    
	}
	
	private String convertSingleNodeToString(TreeNode node) {
		StringBuilder sb = new StringBuilder();
		sb.append(node.getName());
		if (node.hasInformation(TemplatesBasic.ASTNAME)) {
			AstNameInformation ani = (AstNameInformation)node.getInformation(TemplatesBasic.ASTNAME); 
			sb.append("[\"").append(ani.getAstName()).append("\"]");
		}
		if (node instanceof TokenNode) {
			TokenNode tn = (TokenNode)node;
			sb.append("<").append(tn.getStartLine()).append(",")
					.append(tn.getStartCol()).append(",").append(tn.getOffset())
					.append(",\"");
			String text = tn.getToken().getText();
			String replace = escape(text); 
			sb.append(replace).append("\">");
		} else if (node instanceof SplitTreeNode) {
		    SplitTreeNode stn = (SplitTreeNode)node;
		    int stnId = getId(stn);
		    int stnPredId = stn.getPredecessor() == null ? -1 : getId(stn.getPredecessor());
		    int stnSuccId = stn.getSuccessor() == null ? -1 : getId(stn.getSuccessor());
		    sb.append("{").append(stnId).append(",").append(stnPredId)
		            .append(",").append(stnSuccId).append("}");
		    
		}
		return sb.toString();
	}
	
	
	private int getId(SplitTreeNode stn) {
	    Integer id = stnIdMap.get(stn);
	    if (id == null) {
	        id = register(stn);
	    }
	    return id;
	}
	
	private int register(SplitTreeNode stn) {
        int max = 0;
        for (Integer i : stnIdMap.values()) {
            if (i > max) {
                max = i;
            }
        }
        stnIdMap.put(stn, max + 1);
	    return max + 1;
	}
	
	

	public List<String> tokenize(String string) {
		List<String> tokenList = new LinkedList<String>();
		final int DEFAULT = 0;
		final int STRING = 1;
		final int ESCAPE = 2;
		Stack<Integer> stack = new Stack<Integer>();
		stack.push(DEFAULT);
		StringBuilder token = new StringBuilder();
		Character addToNextToken = null;
		boolean tokenEnd = false;
		boolean splitToken = false;
		for (int i = 0; i < string.length(); i++) {
			char c = string.charAt(i);
			int status = stack.peek();
			
			switch(status) {
				case DEFAULT:
					switch (c) {
						case '(':	
							token.append(c);
							tokenEnd = true;
							splitToken = true;
							break;
						case ')':
							token.append(c);
							tokenEnd = true;
							splitToken = true;
							break;
						case ' ':
							tokenEnd = true;
							break;
						case '\"':
							stack.push(STRING);
							addToNextToken = c;
							break;
						default:
							token.append(c);
							break;
					}
					break;
				case STRING:
					switch(c) {
						case '\"':
							token.append(c);
							stack.push(DEFAULT);
							break;
						case '\\':
							token.append(c);
							stack.push(ESCAPE);
							break;
						default:
							token.append(c);
							break;
									
					}
					break;
				case ESCAPE:
					token.append(c);
					stack.pop();
					break;
			}
			
			if (tokenEnd) {
				if (splitToken) {
					String token1 = token.substring(0, token.length() - 1);
					String token2 = Character.toString(token.charAt(token.length() - 1));
					if (token1.length()>0) {
						tokenList.add(token1);
					}
					tokenList.add(token2);
					splitToken = false;
				} else {
					if (token.length()>0) {
						tokenList.add(token.toString());
					}
				}
				token.delete(0, token.length());
				tokenEnd = false;
			}
			if (addToNextToken != null) {
				token.append(addToNextToken);
				addToNextToken = null;
			}
		}
		if (token.length()>0) {
			tokenList.add(token.toString());
		}
		return tokenList;
	}
	
	public TreeNode convertSingleStringToTreeNode(String s) {
		if (s.indexOf("<") > -1 && s.endsWith(">")) {
			return convertSingleStringToTokenNode(s);
		}
		if (s.indexOf("{") > -1 && s.endsWith("}")) {
		    return convertSingleStringToSplitTreeNode(s);
		}
		Pattern p = Pattern.compile("^(.*?)(?:\\[\"(.*?)\"\\])?$");
		Matcher m = p.matcher(s);
		if (m.find()) {
			String name = m.group(1);
			TreeNode tn = new TreeNode(name);
			String astName = m.group(2);
			if (astName != null) {				
				AstNameInformation ani = new AstNameInformation(astName);
				tn.addInformation(ani);
			}
			return tn;
		}
		return null;
	}
	
	public TokenNode convertSingleStringToTokenNode(String s) {
		//(?s) causes the dot to also match line terminators. We need this in order to
		//handle tokens that span multiple lines (comments, C# multiline strings)
		Pattern p = Pattern.compile("(?s)(.*?)<(\\d+?),(\\d+?),(\\d+?),\"(.*)\">");
		Matcher m = p.matcher(s);
		if (m.find()) {
			String name = m.group(1);
			int startLine = Integer.parseInt(m.group(2));
			int startCol = Integer.parseInt(m.group(3));
			int offset = Integer.parseInt(m.group(4));
			String text = m.group(5);
			String replace = unescape(text);
			TokenNode tn = new TokenNode(name, new Token(startLine, startCol, offset, replace));
			return tn;
		}
		return null;
	}
	
    public SplitTreeNode convertSingleStringToSplitTreeNode(String s) {
        Pattern p = Pattern.compile("^(.*?)(?:\\[\"(.*?)\"\\])?\\{(\\d+?),(-?\\d+?),(-?\\d+?)\\}$");
        Matcher m = p.matcher(s);
        if (m.find()) {
            String name = m.group(1);
            SplitTreeNode tn = new SplitTreeNode(name);
            String astName = m.group(2);
            if (astName != null) {              
                AstNameInformation ani = new AstNameInformation(astName);
                tn.addInformation(ani);
            }
            int id = Integer.valueOf(m.group(3));
            int predId = Integer.valueOf(m.group(4));
            int succId = Integer.valueOf(m.group(5));
            
            idStnMap.put(id, tn);
            if (predId != -1) {
                needsPredecessorMap.put(tn, predId);
            }
            if (succId != -1) {
                needsSuccessorMap.put(tn, succId);
            }
            return tn;
        }
        return null;
    }
	
	public TreeNodeRoot convertStringToTree(String serializedTree) {
        stnIdMap.clear();
        needsPredecessorMap.clear();
        needsSuccessorMap.clear();

		List<String> lTokens = tokenize(serializedTree);
		//get and remove first token from list
		String firstToken = lTokens.remove(0);
		Stack<TreeNode> stack = new Stack<TreeNode>();
		TreeNode currentNode = null;
		TreeNode root = null;
		if (!firstToken.equals("(") && !firstToken.equals(")")) {
			currentNode = convertSingleStringToTreeNode(firstToken);
			root = currentNode;
		}
		
		for (String token : lTokens) {
			if (token.equals("(")) {
				stack.push(currentNode);
			} else if (token.equals(")")) {
				stack.pop();
			} else {
				currentNode = convertSingleStringToTreeNode(token);
				stack.peek().addChild(currentNode);
			}
		}
		connectSplitTreeNodes();
		stnIdMap.clear();
		needsPredecessorMap.clear();
		needsSuccessorMap.clear();
		return new TreeNodeRoot(root);
	}
	
	public String escape(String unescapedString) {
		String escaped = unescapedString.replaceAll("\\\\", "\\\\\\\\");
		escaped = escaped.replaceAll("\\\"", "\\\\\\\"");
		return escaped;
	}
	
	public String unescape(String escapedString) {
		String unescaped = escapedString.replaceAll("\\\\\\\"", "\\\"");
		unescaped = unescaped.replaceAll("\\\\\\\\", "\\\\");
		return unescaped;
	}
	
	
	private void connectSplitTreeNodes() {
	    for (SplitTreeNode stn : needsSuccessorMap.keySet()) {
	        int succId = needsSuccessorMap.get(stn);
	        SplitTreeNode stnSucc = idStnMap.get(succId);
	        stn.setSuccessor(stnSucc);
	    }
        for (SplitTreeNode stn : needsPredecessorMap.keySet()) {
            int predId = needsPredecessorMap.get(stn);
            SplitTreeNode stnPred = idStnMap.get(predId);
            stn.setPredecessor(stnPred);
        }
        Set<SplitTreeNode> errorSet = checkSplitTreeNodeConsistency();
        if (!errorSet.isEmpty()) {
            System.out.println(errorSet);
        }
	}
	
	private Set<SplitTreeNode> checkSplitTreeNodeConsistency() {
	    Set<SplitTreeNode> errorSet = new HashSet<SplitTreeNode>();
        for (SplitTreeNode stn : needsSuccessorMap.keySet()) {
            SplitTreeNode stnSucc = stn.getSuccessor();
            if (stnSucc == null) {
                errorSet.add(stn);
            } else {
                SplitTreeNode stnSuccPred = stnSucc.getPredecessor();
                if (stnSuccPred != stn) {
                    errorSet.add(stn);
                    errorSet.add(stnSucc);
                }
            }
        }
        for (SplitTreeNode stn : needsPredecessorMap.keySet()) {
            SplitTreeNode stnPred = stn.getPredecessor();
            if (stnPred == null) {
                errorSet.add(stn);
            } else {
                SplitTreeNode stnPredSucc = stnPred.getSuccessor();
                if (stnPredSucc != stn) {
                    errorSet.add(stn);
                    errorSet.add(stnPred);
                }
            }
        }
        return errorSet;
	}
	
	
	
}
