package org.deft.repository;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.UnsupportedEncodingException;
import java.util.LinkedList;
import java.util.List;

import org.xml.sax.ContentHandler;
import org.xml.sax.ErrorHandler;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;
import org.xml.sax.XMLReader;
import org.xml.sax.helpers.XMLReaderFactory;

/**
 * Base class for all converters that transform xml files to data of any kind using the SAX-Parser
 */
public abstract class XmlToDataConverter {

    public static final int XML_WARNING = 2;
    public static final int XML_ERROR = 1;
    public static final int XML_FATAL_ERROR = 0;

//    private File xmlFile;
    private InputStream fileStream;
    private boolean error = false;
    private XMLReader parser;
    private String schemaFileName;
    private boolean throwUnexpectedExceptions;
    private List<String> xmlErrorMessages = new LinkedList<String>();
    private static final String[] severityString = {"Fatal Error", "Error", "Warning"};

    /**
     * Creates a new XML converter.
     * @param schemaFileName the name of the XML schema file to validate against
     * @param throwUnexpectedExceptions specifies whether exceptions that 
     * occur during
     * the parsing should be passed along or not.
     */
    public XmlToDataConverter(String schemaFileName, boolean throwUnexpectedExceptions) {
        this.schemaFileName = createValidSchemaFileName(schemaFileName);
        this.throwUnexpectedExceptions = throwUnexpectedExceptions;
    }
    
    /**
     * Creates a file name for the XML-Schema file to be used by the XML-Parser. 
     * The input string is converted to a file from which the absolute file 
     * name is computed. As it is required that the file name must have slashes 
     * as separators, all backslashes are replaced by slashes. Furthermore 
     * special characters are escaped (i.e. spaces become %20).
     *
     * @param schemaFileName location of the schema file. 
     * Can be a relative file name.
     * @return the absolute schema file name, uses slashes ('/') as separators
     */
    private String createValidSchemaFileName(String schemaFileName) {
        String ret = new File(schemaFileName).getAbsolutePath();
        ret = ret.replace('\\', '/');
        try {
            ret = java.net.URLEncoder.encode(ret, "UTF-8");
        }
        catch (UnsupportedEncodingException ex) {
            ex.printStackTrace();
        }
        //URLEncoder encoded the colon and the slashes, which is not desired. 
        //Change them back
        ret = ret.replace("%3A", ":");
        ret = ret.replace("%2F", "/");
        //URLEncoder changed spaces into the '+' character, we need, however, %20
        ret = ret.replace("+", "%20");
        return ret;
    }


    /**
     * Returns whether the parsing was successful. This is the case if no errors were found.
     * @return <code>true</code> if there were no errors, otherwise <code>false</code>
     */
    public boolean parsingSuccessful() {
        return !error;
    }

    /**
     * Sets the error flag.
     * @param error the value to be set for error.
     */
    public void setError(boolean error) {
        this.error = error;
    }

    /**
     * Sets a file to be converted and starts the conversion process.
     * @param xmlFile the file to be parsed and converted.
     */
    public void convert(File xmlFile) {
    	try {
			convert(new FileInputStream(xmlFile));
		} catch (FileNotFoundException e) {
			e.printStackTrace();
		}
    }
    
    public void convert(InputStream stream){
    	this.fileStream = stream;
    	if (fileStream != null){
        	convert();
        	try {
				fileStream.close();
			} catch (IOException e) {
			}
    	}

    }


    
    /**
     * Returns the file to be converted.
     * @return the xml file that is converted
     */
//    public File getFile() {
//        return xmlFile;
//    }

    /**
     * Manages the overall conversion process. This includes initialization of data, parser setup
     * and the actual parsing.
     */
    protected void convert() {
        initializeData();
        setupParser();
        parse();
    }

    

    /**
     * Adds an error to the list of encountered XML parsing errors. This method is usually called
     * by XMLErrorHandlers while the file is parsed.
     * @param severity the severity of the error. Can be <code>XML_FATAL_ERROR</code>,
     * <code>XML_ERROR</code> or <code>XML_WARNING</code>
     * @param line the line in which the error was found
     * @param col the column in which the error was found
     * @param message the message string to be stored
     */
    protected void addXmlError(int severity, int line, int col, String message) {
        String s = severityString[severity] + " at line " + line + ", column " + col + ": " + message;
        xmlErrorMessages.add(s);
    }

    /**
     * Returns the list of XML error messages.
     * @return the list of XML error messages
     */
    public List<String> getXmlErrorMessages() {
        return xmlErrorMessages;
    }

    StringBuilder sb = new StringBuilder();


    /**
     * Initializes Data
     */
    protected abstract void initializeData();

    /**
     * Returns the ContentHandler. The ContentHandler is where the parsed file examined and converted
     * to a Java data structure.
     * @return the ContentHandler
     */
    protected abstract ContentHandler getContentHandler();

    /**
     * Returns the ErrorHandler. When an error is found when parsing a file, the ErrorHandler is invoked
     * which in turn can take appropriate measures. Usually the ErrorHandler stores the found errors
     * using {@link #addXmlError(int, int, int, String)}.
     * @return the ErrorHandler
     */
    protected abstract ErrorHandler getErrorHandler();

    /**
     * Sets up the parser. This includes:
     * <ul>
     * <li>making it validate the file against the schema</li>
     * <li>setting the content handler</li>
     * <li>setting the error handler</li>
     * </ul>
     */
    protected void setupParser() {
        error = false;
        xmlErrorMessages.clear();
        try {
        	parser = XMLReaderFactory.createXMLReader();
            parser.setFeature("http://apache.org/xml/features/validation/schema", true);
            parser.setFeature("http://xml.org/sax/features/validation", true);
            parser.setProperty("http://apache.org/xml/properties/schema/external-noNamespaceSchemaLocation",
                schemaFileName);
            parser.setContentHandler(getContentHandler());
            parser.setErrorHandler(getErrorHandler());
        }
        catch (SAXException se) {
            se.printStackTrace();
            error = true;
            if (throwUnexpectedExceptions) {
                //throw new JAVUnexpectedException(se);
           
            }
        }
    }

    /**
     * Starts the actual parsing. Depending on whether a file name or a file have been set for parsing
     * (see {@link #convert(String)} and {@link #convert(File)}) a different start method
     * of the parser is invoked.
     */
    protected void parse() {
        try {
        	parser.parse(new InputSource(fileStream));
            if (parsingSuccessful()) {
                postProcess();
            }
        }
        catch (SAXException se) {
            se.printStackTrace();
            error = true;
            if (throwUnexpectedExceptions) {
                //throw new JAVUnexpectedException(se);
            	
            
            }
        }
        catch (IOException ioe) {
            ioe.printStackTrace();
            error = true;
            if (throwUnexpectedExceptions) {
                //throw new JAVUnexpectedException(ioe);
            	
            	
            }
        }
    }

    /**
     * Called after the parsing has finished successfully. It is intended to be overridden to do
     * arbitrary work that has to be done after the parsing.
     */
    protected void postProcess() {}

}