package org.deft.repository.query;

import java.io.Serializable;

import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Set;

import org.deft.repository.util.XmlUtil;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

public class Query implements Serializable {

	private Document doc;
	private String file;
	private List<String> queryStrings;
	private String attribute;
	
	public Query(Document qDoc) {
		this.doc = qDoc;
        Element root = doc.getDocumentElement();
		this.file = root.getAttribute("file");
		this.queryStrings = createQueryStrings("", root);
	}
	
	public String getFile() {
		return file;
	}
	
    public Document getDocument() {
        return doc;
    }
    
    public String getQueryAttribute(){
    	if (attribute == null){
    		NodeList list = doc.getElementsByTagName("codequery");
    		if (list.getLength() > 0){
    			Element e = (Element)list.item(0);
    			String targetString = e.getAttribute("target");
    			// check if attribute is position
    			if (targetString.contains("[position()=")){
    				attribute = "position";
    			}
    			// check for other attributes
    			else {
    				int start = targetString.lastIndexOf("@");
    				int end = targetString.lastIndexOf("=");
    				if (start > -1 && end > -1 && end > start ){
    					attribute = targetString.substring(start+1, end);
    				}
    			}
    		}
    	}
    	return attribute;
    }
    
	public static Query readQueryFromFile(String fileName) {
        Document queryDoc = XmlUtil.loadXmlFromFile(fileName);
	    return new Query(queryDoc);
	}
	
	public static Query makeQueryFromStrings(String fileName, String... xpaths) {
		Document doc = XmlUtil.makeDocument();
		Element eQuery = doc.createElement("query");
		eQuery.setAttribute("file", fileName);
		for (String path : xpaths) {
			Element eCodeQuery = doc.createElement("codequery");
			eCodeQuery.setAttribute("target", path);
			eQuery.appendChild(eCodeQuery);
		}
		doc.appendChild(eQuery);
		Query query = new Query(doc);
		return query;
	}
	
	public List<String> getQueryStrings() {
		return queryStrings;
	}
	
	private List<String> createQueryStrings(String queryStringStart, Element elem) {
		String newQueryStringStart = append(queryStringStart, elem.getAttribute("target"));
		List queryChildren = getQueryChildren(elem);
		List<String> queryStrings = new LinkedList<String>();
		if (queryChildren.isEmpty()) {
			queryStrings.add(newQueryStringStart);
			return queryStrings;
		}
		for (Object o : queryChildren) {
			Element child = (Element)o;
			queryStrings.addAll(createQueryStrings(newQueryStringStart, child));
		}
		return queryStrings;
	}
    
    private List<Element> getQueryChildren(Element elem) {
        List<Element> queryChildren = new LinkedList<Element>();
        NodeList nl = elem.getChildNodes();
        for (int i = 0; i < nl.getLength(); i++) {
            Node n = (Node)nl.item(i);
            if (n.getNodeType() == Node.ELEMENT_NODE && n.getNodeName().equals("codequery")) {
                queryChildren.add((Element)n);
            }
        }
        return queryChildren;
    }
	
	
	private String append(String queryRoot, String queryNext) {
		if (queryRoot.endsWith("/")) {
			queryRoot = queryRoot.substring(0, queryRoot.length() - 1);
		}
		if (!queryNext.startsWith("/")) {
			queryNext = "/" + queryNext;
		}
		return queryRoot + queryNext;
	}

	public Set<String> getQueryTargets() {
		Set<String> retSet = new HashSet<String>();
		for (String qs : queryStrings) {
			String[] parts = qs.split("/");
			qs = parts[parts.length-1];
			//remove things such as [@name='MyName(string[], int)'] from xpath
			int index = qs.lastIndexOf("[@");
			if (index != -1){
				qs = qs.substring(0,index);
			}
			//remove things such as [2] from xpath
			index = qs.lastIndexOf("[");
			if (index != -1){
				qs = qs.substring(0,index);
			}			
			retSet.add(qs);
		}
		return retSet;
	}
	
    public String toString() {
        StringBuilder sb = new StringBuilder().append(file).append(":");
        for (String s : queryStrings) {
            sb.append("\n");
            sb.append(s);
        }
        return sb.toString();
    }
    
}
