package org.deft.repository.ast.decoration;

import java.util.LinkedList;
import java.util.List;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;

import org.deft.repository.ast.TokenNode;
import org.deft.repository.ast.TreeNode;
import org.deft.repository.util.XmlUtil;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

public class DecoratedTreeToOutputXmlTransformer {

	private TreeNode ast;
	private DecoratorSelection decorators;
    
	public DecoratedTreeToOutputXmlTransformer(TreeNode ast, DecoratorSelection decorators) {
		this.ast = ast;
		this.decorators = decorators;
	}


    /**
     * Converts the syntax tree into an XML output tree. The actual
     * syntax information are lost along the way. The output tree's 
     * structure is made of formatting information, e.g. which tokens
     * belong to which line, which tokens are selected, which tokens
     * are of a special type (keywords, numbers), ...
	 *
     * @return the formatted XML output
     */
    public Document createOutputXml() {
    	if (astConsistsOnlyOfRootNode()) {
    		return createEmptyOutputXmlTree();
    	} else {
    		return createNonEmptyOutputXmlTree();
    	}
    }
    
    private boolean astConsistsOnlyOfRootNode() {
    	return !ast.hasChildren();
    }
    
    	
	private Document createEmptyOutputXmlTree() {
		Document doc = createDocument();
		Element sourceCode = doc.createElement("sourcecode");
		doc.appendChild(sourceCode);
		return doc;
	}
	
	private Document createDocument() {
		try {
			DocumentBuilder builder = DocumentBuilderFactory.newInstance().newDocumentBuilder();
			Document doc = builder.newDocument();
			return doc;

		} catch (ParserConfigurationException e) {
			e.printStackTrace();
		}
		return null;
	}
	
	private Document createNonEmptyOutputXmlTree() {
		Document xmlTree = createEmptyOutputXmlTree();
		List<GroupList> groupLists = setupAndCollectGroupLists();
		SubGroupTreeBuilder treeBuilder = createAndSetupSubGroupTreeBuilder(groupLists);
		SubGroupNode root = treeBuilder.getSubGroupNodeRoot();
		root.makeXml(xmlTree.getDocumentElement());
        removeDummyNodes(xmlTree);
        return xmlTree;	
	}
	

	private List<GroupList> setupAndCollectGroupLists() {
		setupTokenNodeGroups();
		List<GroupList> groupLists = collectGroupLists();
		return groupLists;
	}

	
	private void setupTokenNodeGroups() {
		List<TokenNode> tokenNodes = ast.serialize();
		for (AstDecorator decorator : decorators) {
			decorator.buildAndAddGroupLists(tokenNodes);
		}
	}
	


	private List<GroupList> collectGroupLists() {
		List<GroupList> groupLists = new LinkedList<GroupList>();
		for (AstDecorator decorator : decorators) {
			groupLists.addAll(decorator.getGroupLists());
		}
		return groupLists;
	}
		

    
    private SubGroupTreeBuilder createAndSetupSubGroupTreeBuilder(List<GroupList> groupLists) {
        List<TokenNode> tokenNodes = ast.serialize();
    	SubGroupTreeBuilder treeBuilder = new SubGroupTreeBuilder(tokenNodes);
    	for (GroupList groupList : groupLists) {
    		treeBuilder.addGroupListAndSplitGroups(groupList);
    	}
    	return treeBuilder;
    }
    



	
	public void removeDummyNodes(Document doc) {
		Element root = doc.getDocumentElement();
		removeDummiesFromNode(root);
	}
	
	private void removeDummiesFromNode(Element parent) {
		org.w3c.dom.NodeList dummyList = XmlUtil.evaluateXpath(parent, "_dummy_");
		if (dummyList.getLength() > 0) {
			for (int i = 0; i < dummyList.getLength(); i++) {
				Element dummyElement = (Element)dummyList.item(i);
				org.w3c.dom.NodeList nl = dummyElement.getChildNodes();
				while (nl.getLength() > 0) {
					Node n = nl.item(0);
					dummyElement.removeChild(n);
					parent.insertBefore(n, dummyElement);
				}
				parent.removeChild(dummyElement);
			}
			removeDummiesFromNode(parent);
		} else {
			NodeList children = parent.getChildNodes();
			for (int i = 0; i < children.getLength(); i++) {
				Node node = children.item(i);
				if (node instanceof Element) {
					removeDummiesFromNode((Element)node);
				}
			}
		}
	}


}
