package org.deft.repository.ast.decoration;

import java.util.HashMap;

import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.UUID;

import org.deft.repository.Util;


/**
 * This class represents a code snippet format. It contains a name, a number of
 * targets this format can be applied on, a list of hide expression that
 * specify which node elements shall be hidden and a number of display styles
 * that are allowed with this format.
 * 
 * @author Thomas.Janke
 * 
 */
public class Format {
	
	public enum DisplayType {
		INLINE, BLOCK;
		
		@Override
		public String toString() {
			return super.toString().toLowerCase();
		}
		
		public static DisplayType fromString(String type){
			for (DisplayType t : values()) {
				if(t.toString().equals(type)){
					return t;
				}
			}
			return null;
		}
	}
	
	
	
	
	private String name;
	private List<String> displayList = new LinkedList<String>();
	//Map<target, List<ReplaceGroup>
	private Map<String, List<ReplaceGroup>> replaceMap = new HashMap<String, List<ReplaceGroup>>();
	
	public Format(String name) {
		this.name = name;
	}
	
	public String getName() {
		return name;
	}
	
	
	/**
	 * Used for adding targets without hide statements (e.g. target "all")
	 * @param target
	 */
	public void addTarget(String target) {
		if (!replaceMap.containsKey(target)) {
			List<ReplaceGroup> lRg = new LinkedList<ReplaceGroup>();
			replaceMap.put(target, lRg);
		}
	}
	
	private String createId(List<ReplaceGroup> lRg) {
		String id = UUID.randomUUID().toString();
		boolean existingIdFound = true;
		while (existingIdFound) {
			existingIdFound = false;
			for (ReplaceGroup rg : lRg) {
				if (rg.getId().equals(id)) {
					existingIdFound = true;
					id = UUID.randomUUID().toString();
				}
			}
		}
		return id;
	}
	
	public void addReplaceAll(String target, List<String> xPaths, String replace) {
		List<ReplaceGroup> lRg = Util.lazyGet(replaceMap, target);
		String id = createId(lRg);
		//cut trailing slash
		if (target.endsWith("/")) {
			target = target.substring(0, target.length() - 1);
		}

		//if ReplaceGroup with id already exists, remove it
		for (ReplaceGroup rg : lRg) {
			if (rg.getId().equals(id)) {
				lRg.remove(rg);
			}
		}
		ReplaceGroup rg = new ReplaceGroup(id, xPaths, null, replace);
		lRg.add(rg);
	}
	
	
	public void addReplaceExcept(String target, List<String> xPaths, String replace) {
		List<ReplaceGroup> lRg = Util.lazyGet(replaceMap, target);
		String id = createId(lRg);
		//cut trailing slash
		if (target.endsWith("/")) {
			target = target.substring(0, target.length() - 1);
		}

		//if ReplaceGroup with id already exists, remove it
		for (ReplaceGroup rg : lRg) {
			if (rg.getId().equals(id)) {
				lRg.remove(rg);
			}
		}
		ReplaceGroup rg = new ReplaceGroup(id, null, xPaths, replace);
		lRg.add(rg);
	}
	
	public String getHideReplacement(String target, String xPath) {
		for (ReplaceGroup rg : getReplaceGroups(target)) {
			if (rg.isHidden(xPath)) {
				return rg.getReplaceString();
			}
		}
		return null;
	}
	
	public boolean isHidden(String target, String xPath) {
		for (ReplaceGroup rg : getReplaceGroups(target)) {
			if (rg.isHidden(xPath)) {
				return true;
			}
		}
		return false;
	}
	
	public boolean validForTarget(String target) {
		return replaceMap.containsKey(target);
	}	
	
	public List<ReplaceGroup> getReplaceGroups(String target) {
		List<ReplaceGroup> list = replaceMap.get(target);
		if (list == null) {
			return new LinkedList<ReplaceGroup>();
		} else {
			return new LinkedList<ReplaceGroup>(list);
		}
	}
	
	public ReplaceGroup getReplaceGroup(String target, String replaceGroupId) {
		for (ReplaceGroup rg : getReplaceGroups(target)) {
			if (rg.getId().equals(replaceGroupId)) {
				return rg;
			}
		}
		return null;
	}
	
	/**
	 * Returns for a hide string a list of hide strings that should
	 * be replaced with the same target
	 * 
	 * @param target
	 * @param hideString
	 * @return
	 */
	/*public List<String> getReplaceGroup(String target, String hideString) {
	
	}*/
	
	/*public boolean replacemenEmpty() {
		return replaceMap.size() == 0;
	}*/
	
	
	
	public List<String> getTargetList() {
		return new LinkedList<String>(replaceMap.keySet());
	}
	
	public void addDisplay(String display) {
		if (!displayList.contains(display)) {
			displayList.add(display);
		}
	}
	
	public void addDisplayList(String[] display) {
		for (String string : display) {
			addDisplay(string);
		}
	}
	
	public List<String> getDisplayList() {
		return displayList;
	}	
	
	public boolean validForDisplay(String currentDisplayType) {
		return displayList.contains(currentDisplayType);
	}
	
	public String toString() {
		return "Format " + name + ": " + replaceMap;
	}


	@Override
	public boolean equals(Object obj) {
		if (obj instanceof Format){
			Format f = (Format)obj;
			return f.getName().equals(getName());
		}
		return false;
	}


}
//	
//	private String name;
//	private List<String> displayList;
//	private Map<String, Map<String, String>> hideMap;
//	private Map<String, Map<String, String>> hideExceptMap;
//	
//	public Format(String name) {
//		this.name = name;
//		this.hideMap = new LinkedHashMap<String, Map<String, String>>();
//		this.hideExceptMap = new LinkedHashMap<String, Map<String, String>>();
//		this.displayList = new LinkedList<String>();
//	}
//	
//	public String getName() {
//		return name;
//	}
//	
//	public boolean isHidden(String target, String xPath) {
//		if (target.equals(".")) {
//			for (Map<String, String> targetMap : hideMap.values()) {
//				if (targetMap.keySet().contains(xPath)) {
//					return true;
//				}
//			}
//			return false;
//		} else {
//			Map<String, String> mapForTarget = hideMap.get(target);
//			if (mapForTarget == null){
//				return false;
//			}
//			if (!mapForTarget.containsKey(".") && !mapForTarget.containsKey(xPath)){
//				return false;
//			}
//			return true;
//		}
//	}
//	
//	public List<String> getHideList(String target) {
//		List<String> hideList = new LinkedList<String>();	
//		if (target.equals(".")) {
//			for (Map<String, String> targetMap : hideMap.values()) {
//				hideList.addAll(targetMap.keySet());
//			}
//			return hideList;
//		} else {
//			//if target a normal name return all xpaths for that name
//			Map<String, String> mapForTarget = hideMap.get(target);
//			if (mapForTarget != null) {
//				hideList.addAll(mapForTarget.keySet());
//			}
//			return hideList;
//		}		
//	}	
//	
//	public String getHideReplacement(String target, String xPath){
//		if (target.equals(".")) {
//			for (Map<String, String> targetMap : hideMap.values()) {
//				if (targetMap.keySet().contains(xPath)) {
//					return targetMap.get(xPath);
//				}
//			}
//			return null;
//		} else {
//			Map<String, String> mapForTarget = hideMap.get(target);
//			if (mapForTarget == null){
//				return null;
//			}
//			
//			if (mapForTarget.containsKey(xPath)){
//				return mapForTarget.get(xPath);
//			}
//			
//			return mapForTarget.get(".");
//		}
//	}
//	
//	/**
//	 * Used for adding targets without hide statements (e.g. target "all")
//	 * @param target
//	 */
//	public void addTarget(String target) {
//		Map<String, String> mapForTarget = hideMap.get(target);
//		if (mapForTarget == null) {
//			mapForTarget = new LinkedHashMap<String, String>();
//			hideMap.put(target, mapForTarget);
//		}		
//	}
//	
//	
//	public void addHideStatement(String target, String xPath, String replace) {
//		Map<String, String> mapForTarget = hideMap.get(target);
//		if (mapForTarget == null) {
//			mapForTarget = new LinkedHashMap<String, String>();
//			hideMap.put(target, mapForTarget);
//		}
//		//if the string object "replace" already exists, 
//		//create new object with same content
//		if (replace != null){
//			for (String v : mapForTarget.values()) {
//				if (v == replace) {
//					replace = new String(replace);
//				}
//			}
//		}
//		mapForTarget.put(xPath, replace);
//	}
//
//	
//	public void addHideGroup(String target, List<String> hidePaths,
//			List<String> hideExceptPaths, String replace) {
//		Map<String, String> mapForTarget = hideMap.get(target);
//		if (mapForTarget == null) {
//			mapForTarget = new LinkedHashMap<String, String>();
//			hideMap.put(target, mapForTarget);
//		}
//		//if the string object "replace" already exists, 
//		//create new object with same content
//		for (String v : mapForTarget.values()) {
//			if (v != null && v == replace) {
//				replace = new String(replace);
//			}
//		}
//		for (String path : hidePaths) {
//			mapForTarget.put(path, replace);
//		}
//	}
//	
//	public boolean validForTarget(String target) {
//		for (String string : hideMap.keySet()) {
//			if (string.equals(target) || string.equals("*")) {
//				return true;
//			}
//		}
//		return false;
//	}	
//	
//	/**
//	 * Returns for a hide string a list of hide strings that should
//	 * be replaced with the same target
//	 * 
//	 * @param target
//	 * @param hideString
//	 * @return
//	 */
//	public List<String> getReplaceGroup(String target, String hideString) {
//		List<String> groupList = new LinkedList<String>();
//		String replacement = getHideReplacement(target, hideString);
//		//added by thomas: i uncommented this because it can of course be, that there is no replacement defined
//		//and therefor it is null
//		//if (replacement != null) {
//			Map<String, String> mapForTarget = hideMap.get(target);
//			for (String h : mapForTarget.keySet()) {
//				//note the string comparison with == here
//				//we want all hide strings that map to the 
//				//same replacement string OBJECT
//				if (mapForTarget.get(h) == replacement) {
//					groupList.add(h);
//				}
//			}
//		//}
//		return groupList;
//	}
//	
//	public boolean hideListEmpty() {
//		return this.hideMap.size() == 0;
//	}
//	
//	
//	
//	public List<String> getTargetList() {
//		return new LinkedList<String>(hideMap.keySet());
//	}
//	
//	public void addDisplay(String display) {
//		if (!displayList.contains(display)) {
//			displayList.add(display);
//		}
//	}
//
//	public void addDisplayList(String[] display) {
//		for (String string : display) {
//			addDisplay(string);
//		}
//	}
//	
//	public List<String> getDisplayList() {
//		return displayList;
//	}	
//	
//	public boolean validForDisplay(String currentDisplayType) {
//		return displayList.contains(currentDisplayType);
//	}
//	
//	public String toString() {
//		return "Format " + name + ": " + hideMap;
//	}
//
////	private String name;
////	private List<String> displayList;
////	private HashMap<String,String> hideMap;
////
////	private List<String> targetList;
////
////	public Format(String name) {
////		this.name = name;
////		this.displayList = new ArrayList<String>();
////		this.hideMap = new HashMap<String,String>();
////		this.targetList = new ArrayList<String>();
////	}
////
////	public String getName() {
////		return this.name;
////	}	
////
////	public List<String> getTargetList() {
////		return targetList;
////	}
////	
////	public void addTargetList(String[] targets) {
////		for (String string : targets) {
////			addTarget(string);
////		}
////	}
////
////	public void addTarget(String target) {
////		if (!targetList.contains(target)) {
////			targetList.add(target);
////		}
////	}
////
////
////	public void addDisplay(String display) {
////		if (!displayList.contains(display))
////			displayList.add(display);
////	}
////
////	public void addHideStatementMap(HashMap<String,String> hideMap) {
////		this.hideMap.putAll(hideMap);
////	}
////
////	public void addHideStatement(String hide, String replace) {
////		if (!hideMap.containsKey(hide)){
////			
////			if(replace == null){
////				replace = new String();
////			}
////			hideMap.put(hide,replace);
////			
////		}
////			
////	}
////
////	/**
////	 * Returns true if there are no hide expressions defined
////	 * 
////	 * @return
////	 */
////	public boolean hideListEmpty() {
////		return this.hideMap.size() == 0;
////	}
////
////	/**
////	 * Returns true if the given xpath expression matches a hidden expression
////	 * defined for that format
////	 * 
////	 * @param xpath
////	 * @return
////	 */
////	public boolean isHidden(String path) {
////		String hide = getMatchingHide(path);
////		return hide != null;
////	}
////	
////	public String getHideReplacement(String path){
////		String hide = getMatchingHide(path);
////		if (hide != null){
////			return hideMap.get(hide);
////		}
////		return null;
////	}
////	
////	public Map<String, String> getHideMap() {
////		return hideMap;
////	}
////	
////	public List<String> getDisplayList(){
////		return displayList;
////	}
////	
////	public List<String> getHideList() {
////		return new LinkedList<String>(hideMap.keySet());
////	}
////	
////	private String getMatchingHide(String path){
////		
////		if (hideMap.size() == 0) {
////			return null;
////		}
////		
////		for (String string : hideMap.keySet()) {
////
////			String[] hideParts = string.split("/");
////			String[] pathParts = path.split("/");
////			
////			if (hideParts.length > pathParts.length){
////				continue;
////			}
////			
////			boolean match = true;
////			for (int i = 0; i < hideParts.length; i++) {
////				String hidePart = hideParts[i];
////				String pathPart = pathParts[i];
////
////				if (!hidePart.equals(pathPart)) {
////					if (!endsWithCounter(hidePart)) {
////						if (!hidePart.equals(removeCounter(pathPart))) {
////							match = false;
////							break;
////						}
////					} else {
////						match = false;
////						break;
////					}
////				}
////			}
////			if (match){
////				return string;
////			}
////		}
////		return null;
////	}
////
//////	private String getMatchingHide(String path){
//////		
//////		if (hideMap.size() == 0) {
//////			return null;
//////		}
//////		
//////		for (String string : hideMap.keySet()) {
//////
//////			String[] hideParts = string.split("/");
//////			String[] pathParts = path.split("/");
//////			if (hideParts.length != pathParts.length) {
//////				continue;
//////			}
//////			boolean match = true;
//////			for (int i = hideParts.length - 1; i >= 0; i--) {
//////				String h = hideParts[i];
//////				String p = pathParts[i];
//////
//////				if (!h.equals(p)) {
//////					if (!endsWithCounter(h)) {
//////						if (!h.equals(removeCounter(p))) {
//////							match = false;
//////							break;
//////						}
//////					} else {
//////						match = false;
//////						break;
//////					}
//////				}
//////			}
//////			if (match){
//////				return string;
//////			}
//////		}
//////		return null;
//////	}
////	
////	
////	/**
////	 * Returns true if the format can be applied on the target
////	 * 
////	 * @param currentDisplayType
////	 * @return
////	 */
////	public boolean validForTarget(String target) {
////		for (String string : targetList) {
////			if (string.equals(target) || string.equals("*")) {
////				return true;
////			}
////		}
////		return false;
////	}
////
////	/**
////	 * Returns true if the format can be applied on the given display type
////	 * 
////	 * @param currentDisplayType
////	 * @return
////	 */
////	public boolean validForDisplay(String currentDisplayType) {
////		return displayList.contains(currentDisplayType);
////	}
////
////	private boolean endsWithCounter(String part) {
////		int i = part.lastIndexOf("[");
////		if (i == -1) {
////			return false;
////		}
////		return true;
////	}
////
////	private String removeCounter(String part) {
////		return part.substring(0, part.lastIndexOf("["));
////	}
////	
////	public String toString() {
////		return "Format " + getName();
////	}
//	
//	@Override
//	public boolean equals(Object obj) {
//		if (obj instanceof Format){
//			Format f = (Format)obj;
//			return f.getName().equals(getName());
//		}
//		return false;
//	}
//
//}

