package org.deft.repository.ast.decoration;

import java.util.LinkedList;
import java.util.List;

import org.deft.repository.ast.TokenNode;
import org.w3c.dom.Document;
import org.w3c.dom.Element;


/**
 * <p>Tokens can be grouped together in Groups by several means. An example 
 * would be groups of selected or unselected tokens (see {@link SelectedGroup}). 
 * Another possibility is to group all TokenNodes that contain Tokens which 
 * belong to the same line in the source code file (see {@LineGroup}).</p> 
 *  
 * <p>A Group does not contain the TokenNodes directly. Instead it contains a List 
 * of {@link SubGroup SubGroups}, which hold the actual TokenNodes. This is because 
 * otherwise it would be difficult to combine several types of groups.</p>
 * 
 * <p>Imagine, for example, that the final XML-output should contain markup for both 
 * selected/unselected groups and  lines (i.e. tokens that belong to the same code 
 * line are grouped together). As a selected or unselected group can  potentially 
 * start or stop in the middle of a line, the groups could overlap:</p>
 * 
 * <p>group by selection:</p>
 * 
 * <pre>
 * <selected>public static void</selected><unselected> main(String[] args) {
 *     int i</unselected><selected> = 0;</selected>
 *     ...
 * </pre>
 * 
 * <p>group by lines:</p>
 * 
 * <pre>
 * <line linenumber="1">public static void main(String[] args) {</line>
 * <line linenumber="2">    int i = 0;</line>
 *     ...
 * </pre> 
 *  
 * <p>Overlapping groups can, however, not be represented in the XML-output. 
 * The solution is to split groups.  If the selected/unselected groups are 
 * considered "top level", the line groups will have to be split:</p> 
 *
 * <pre>
 * <selected><line linenumber="1">public static void</line></selected><unselected><line linenumber="1"> main(String[] args) {</line>
 * <line linenumber="2">    int i</line></unselected><selected><line linenumber="2"> = 0;</line></selected>
 *     ...
 * </pre>
 *
 * <p>The SubGroups are used to represent such "split parts".</p>
 * 
 * @author Andreas Bartho
 */
public abstract class Group {
    
    private List<SubGroup> subGroupList = new LinkedList<SubGroup>();

    /**
     * <p>Creates a Group. The Group contains exactly 1 SubGroup with the
     * position information <code>startLine</code>, <code>startCol</code> 
     * and <code>offset</code>, which are set to <code>-1</code>. 
     * These are invalid position markers, but they will
     * be automatically updated internally whenever a new TokenNode is added.</p>
     *
     * <p>This is the standard constructor for most Groups.</p>
     */
    public Group() {
        this(-1, -1, -1);
    }
    
    /**
     * <p>Creates a new Group with exactly 1 SubGroup. The SubGroup will be
     * created using the passed parameters.</p>
     * 
     * <p>This is the constructor for empty Groups. As soon
     * as the first TokenNode is added to the Group, the position information
     * will be computed according to the position information of the TokenNodes'
     * tokens.</p>
     * 
     * @param startLine the line on which the Group starts
     * @param startCol the column on which the Group starts
     * @param offset the position on which the Group starts
     */
    public Group(int startLine, int startCol, int offset) {
        SubGroup initialSubGroup = new SubGroup(true, true, startLine, startCol, offset);
        subGroupList.add(initialSubGroup);
    }
    
    /**
     * <p>Splits a Group after the desired position by splitting the
     * corresponding SubGroup.</p> 
     * 
     * <p>The SubGroup that contains the TokenNode at the position specified by
     * <code>tokenNodeIndex</code> is replaced by 2 SubGroups. All TokenNodes
     * of the SubGroup with an index that is less than or equal <code>tokenNodeIndex</code>
     * are put into the first new SubGroup, the remaining TokenNodes are put into
     * the second new SubGroup. If the <code>tokenNodeIndex</code> equals the size
     * of the SubGroup - 1 (i.e. it points at the last TokenNode), then no
     * split needs to be performed.</p>
     * 
     * <p>Example:</p>
     * 
     * <pre>
     * before split:
     * -------------
     * 
     * 2 SubGroups, tokenNodeIndex is 5
     * 
     * tokenNodeIndex here:                            |
     *                                                 v
     * SubGroups with TokenNodes: [a   b   c] [d   e   f   g]
     * positions of TokenNodes:    0   1   2   3   4   5   6
     * 
     * 
     * after split:
     * ------------
     * 
     * 3 SubGroups
     * 
     * SubGroups with TokenNodes: [a   b   c] [d   e   f] [g]
     * positions of TokenNodes:    0   1   2   3   4   5   6 
     * </pre>
     * 
     * @param tokenNodeIndex the position to split the Group at 
     */
    public void split(int tokenNodeIndex) {
        //get the SubGroup to split
        int subGroupIndex = getSubGroupIndexForTokenNodeIndex(tokenNodeIndex);
        SubGroup sg = subGroupList.get(subGroupIndex);
        //firstToken cannot be null, because sg cannot be empty
        TokenNode firstToken = sg.getFirstTokenNode();
        //index of the first TokenNode of the SubGroup to be split
        int sgTokenNodeStart = getTokenNodeList().indexOf(firstToken);
        
        if (tokenNodeIndex != getTokenNodeList().size() - 1) {
	        SubGroup sg1 = new SubGroup(sg.isStart(), false, -1, -1, -1);
	        //for (int i = 0; i <= tokenNodeIndex - sgTokenNodeStart; i++) {
	        	//sg.getTokenNodeList().subList(fromIndex, toIndex)
	            //sg1.addTokenNode(sg.getTokenNodeList().get(i));
	        sg1.addTokenNodes(sg.getTokenNodeList()
	        		.subList(0, tokenNodeIndex - sgTokenNodeStart + 1));
	        //}
	        SubGroup sg2 = new SubGroup(false, sg.isEnd(), -1, -1, -1);
	        /*for (int i = tokenNodeIndex - sgTokenNodeStart + 1; i < sg.getTokenNodeCount(); i++) {
	            sg2.addTokenNode(sg.getTokenNodeList().get(i));
	        }*/
	        sg2.addTokenNodes(sg.getTokenNodeList()
	        		.subList(tokenNodeIndex - sgTokenNodeStart + 1, sg.getTokenNodeCount()));
	        subGroupList.remove(sg);
	        subGroupList.add(subGroupIndex, sg1);
	        subGroupList.add(subGroupIndex + 1, sg2);
        }
    }
    
    /**
     * Returns the SubGroup at position <code>subGroupIndex</code>. 
     *
     * @param subGroupIndex the position of the SubGroup within this Group
     * @return the SubGroup at the requested position
     */
    public SubGroup getSubGroup(int subGroupIndex) {
        return subGroupList.get(subGroupIndex);
    }
    
    /**
     * <p>Returns a List of all SubGroups of this Group.</p>
     * 
     * <p>Note that not the actual List field of the Group
     * object is returned but only a copy. The SubGroups in that copied list
     * are, however, the original SubGroup objects.</p>
     *  
     * @return the List of SubGroups
     */
    public List<SubGroup> getSubGroupList() {
        return new LinkedList<SubGroup>(subGroupList);
    }
    
    /**
     * <p>Returns a list of all TokenNodes of this Group.</p>
     * 
     * @return a list of TokenNodes
     */    
    public List<TokenNode> getTokenNodeList() {
        List<TokenNode> list = new LinkedList<TokenNode>();
        for (SubGroup sg : subGroupList) {
            list.addAll(sg.getTokenNodeList());
        }
        return list;
    }
    

    /**
     * Returns the position of SubGroup that contains the TokenNode
     * at position <code>tokenNodeInde</code>.
     * 
     * @param tokenNodeIndex the position of the TokenNode for which the
     * SubGroup is to be returned
     * @throws IllegalArgumentException if <code>tokenNodeIndex</code> 
     * is less than 0 or greater or equal than the number of Tokens in the group.
     * 
     * @return the position of the SubGroup containing the specified TokenNode
     */
    public int getSubGroupIndexForTokenNodeIndex(int tokenNodeIndex) {
        if (tokenNodeIndex < 0 || tokenNodeIndex >= getTokenNodeCount()) {
            throw new IllegalArgumentException("Position must be >= 0 and < "
                    + getTokenNodeCount() + " (the number of tokens in the SubGroup), "
                    + "but it was " + tokenNodeIndex);
        }
        
        int skippedLength = 0;
        for (int i = 0; i < subGroupList.size(); i++) {
            SubGroup sg = subGroupList.get(i);
            if (skippedLength + sg.getTokenNodeCount() > tokenNodeIndex) {
                return i;
            }
            skippedLength += sg.getTokenNodeCount();
        }
        //never executed
        return -1;    
    }
    
    /**
     * Returns the first SubGroup of the Group.
     * 
     * @return the first SubGroup
     */
    public SubGroup getFirstSubGroup() {
    	List<SubGroup> sgList = getSubGroupList();
    	return sgList.get(0);
    }
    
    /**
     * Returns the last SubGroup of the Group.
     * 
     * @return the last SubGroup
     */
    public SubGroup getLastSubGroup() {
    	List<SubGroup> sgList = getSubGroupList();
    	return sgList.get(sgList.size() - 1);
    }    
    
    /**
     * <p>Returns the first TokenNode of the Group</p>
     * 
     * <p>If the Group is empty, <code>null</code> is returned.</p>
     * 
     * @return the first TokenNode or <code>null</code> if the
     * Group is empty
     */
    public TokenNode getFirstTokenNode() {
        return getFirstSubGroup().getFirstTokenNode();
    }
    
    /**
     * <p>Returns the last TokenNode of the Group</p>
     * 
     * <p>If the Group is empty, <code>null</code> is returned.</p>
     * 
     * @return the last TokenNode or <code>null</code> if the
     * Group is empty
     */       
    public TokenNode getLastTokenNode() {
        return getLastSubGroup().getLastTokenNode();
    }  
    
    /**
     * <p>Adds a TokenNode to the Group.</p>
     * 
     * <p>Adding a TokenNode is usually only done directly
     * after the creation of the SubGroup. Note that TokenNodes
     * are expected to be added in the same order 
     * as they appear in the source code. No automatic sorting
     * is performed and no checks for left-out TokenNodes are
     * done.</p>
     * 
     * <p>The TokenNode is added to the last SubGroup of the Group</p>
     * 
     * @param node the TokenNode to be added
     */
    public void addTokenNode(TokenNode node) {
        getLastSubGroup().addTokenNode(node);
    } 
    
    /**
     * Returns the number of TokenNodes in the Group.
     * 
     * @return the number of TokenNodes
     */
    public int getTokenNodeCount() {
        int size = 0;
        for (SubGroup sg : subGroupList) {
            size += sg.getTokenNodeCount();
        }        
        return size;
    }
    
    /**
     * Returns whether this Group is empty, i.e.
     * does not contain any TokenNodes.
     * 
     * @return <code>true</code> if there are no TokenNodes
     * in the Group, otherwise <code>false</code>;
     */    
    public boolean isEmpty() {
        return getTokenNodeCount() == 0; 
    }
    
    /**
     * Returns the number of SubGroups in the Group.
     * 
     * @return the number of SubGroups
     */
    public int getSubGroupCount() {
        return subGroupList.size();
    }
    
    /**
     * Returns the column on which the Group ends.
     * 
     * @return the column on which the Group ends
     */
    public int getEndCol() {
        return getLastSubGroup().getEndCol();
    }

    /**
     * Returns the line on which the Group ends.
     * 
     * @return the line on which the Group ends
     */
    public int getEndLine() {
        return getLastSubGroup().getEndLine();
    }

    /**
     * Returns the length of the Group, i.e. the number
     * of characters that it spans in the source code.
     * 
     * @return the length of the Group
     */    
    public int getLength() {
        return getLastSubGroup().getOffset() + getLastSubGroup().getLength() 
                - getFirstSubGroup().getOffset();
    }

    /**
     * Returns the start position of the Group, i.e.
     * the index of the byte at which it begins in the source code.
     * 
     * @return the start offset of the Group
     */
    public int getOffset() {
        return getFirstSubGroup().getOffset();
    }

    /**
     * Returns the column on which the Group starts.
     * 
     * @return the column on which the Group starts
     */
    public int getStartCol() {
        return getFirstSubGroup().getStartCol();
    }

    /**
     * Returns the line on which the Group starts.
     * 
     * @return the line on which the Group starts
     */    
    public int getStartLine() {
        return getFirstSubGroup().getStartLine();
    }    
    
    

    
    /**
     * Creates the XML element to represent this Group in the XML output tree.
     * Only the plain element must be created, no children are allowed, especially
     * not the XML representations of the TokenNodes. 
     * 
     * @param doc the Document to which Element belongs. Needed to create
     * the Element
     * 
     * @return the XML element representing this Group
     */
    protected abstract Element createXmlTag(Document doc);
    
    public Element makeXmlTag(Document doc) {
    	return createXmlTag(doc);
    }
    

    public String toString() {
        StringBuilder sb = new StringBuilder();
        sb.append("[");
        for (int i = 0; i < subGroupList.size(); i++) {
            sb.append(subGroupList.get(i).toString());
            if(i < subGroupList.size() - 1) {
                sb.append(",");
            }
        }
        sb.append("]");
        return sb.toString();
    }

    public abstract boolean isDummy();
}
