package org.deft.repository.ast.decoration;

import java.util.LinkedList;
import java.util.List;

import org.deft.repository.ast.Token;
import org.deft.repository.ast.TokenNode;
import org.w3c.dom.Document;
import org.w3c.dom.Element;



/**
 * <p>A SubGroup is a helper class that is necessary to set 
 * up a hierarchy between different types of Groups.</p>
 * 
 * <p>Usually SubGroups contain at least 1 TokenNode. The only 
 * exception is, when the containing Group is empty itself.
 * (e.g. a LineGroup denoting an empty line)</p>
 * 
 * See {@link Group} for details.
 * 
 * @author Andreas Bartho
 */
public class SubGroup {
    

    //Sometimes Groups have to be split. This is done by dividing
    //its TokenNodes into multiple SubGroups. The fields
    //<code>isStart</code> and <code>isEnd</code> indicate
    //whether the SubGroup is the first and/or last SubGroup
    //of its Group. 
    private boolean isStart, isEnd;
    
    
    //The List of TokenNodes that this SubGroup contains.
    private List<TokenNode> nodeList = new LinkedList<TokenNode>();
    
    //A SubGroup as position information, just as a TokenNode has.
    //Usually these position information are computed from the
    //Tokens in the TokenNodes. If the SubGroup is empty, however, 
    //these fields can also be set manually (via the constructor)
    private int startLine = -1;
    private int startCol = -1;
    private int endLine = -1;
    private int endCol = -1;
    private int offset = -1;
    private int length = -1;
    
    /**
     * <p>Creates a new SubGroup.</p>
     * 
     * <p>This constructor should not be used if this SubGroup is
     * meant to remain empty. For such cases use 
     * {@link @SubGroup(boolean, boolean, int, int, int}.</p>
     * 
     * @param isStart must be <code>true</code> if this is the first
     * SubGroup of the containing Group, otherwise <code>false</code>.
     * @param isEnd must be <code>true</code> if this is the last
     * SubGroup of the containing Group, otherwise <code>false</code>.
     */
    public SubGroup(boolean isStart, boolean isEnd) {
        this(isStart, isEnd, -1, -1, -1);
    }
    
    /**
     * <p>Creates a new SubGroup with some status and position information.</p>
     * 
     * <p>The fields startLine, startCol and offset are position information for
     * this SubGroup. By default a SubGroup's position is computed from the
     * Tokens of the TokenNodes it contains. If the SubGroup remains empty, 
     * however, its position is specified by the remaining 3 int parameters.</p>  
     * 
     * @param isStart must be <code>true</code> if this is the first
     * SubGroup of the containing Group, otherwise <code>false</code>.
     * @param isEnd must be <code>true</code> if this is the last
     * SubGroup of the containing Group, otherwise <code>false</code>.
     * @param startLine the line on which the SubGroup starts
     * @param startCol the column on which the SubGroup starts
     * @param offset the position on which the SubGroup starts
     */
    public SubGroup(boolean isStart, boolean isEnd, int startLine, int startCol, int offset) {
        this.isStart = isStart;
        this.isEnd = isEnd;
        this.startLine = startLine;
        this.endLine = startLine;
        this.startCol = startCol;
        this.endCol = startCol;
        this.offset = offset;
        this.length = 0;
    }

    /**
     * Returns whether this is the last SubGroup of the Group.
     * 
     * @return <code>true</code>, if it is the last SubGroup,
     * otherwise <code>false</code>.
     */
    public boolean isEnd() {
        return isEnd;
    }

    /**
     * Returns whether this is the first SubGroup of the Group.
     * 
     * @return <code>true</code>, if it is the first SubGroup,
     * otherwise <code>false</code>.
     */    
    public boolean isStart() {
        return isStart;
    }
    
    /**
     * Returns the column on which the SubGroup ends.
     * 
     * @return the column on which the SubGroup ends
     */
    public int getEndCol() {
        return endCol;
    }

    /**
     * Returns the line on which the SubGroup ends.
     * 
     * @return the line on which the SubGroup ends
     */    
    public int getEndLine() {
        return endLine;
    }

    /**
     * Returns the length of the SubGroup, i.e. the number
     * of characters that it spans in the source code.
     * 
     * @return the length of the SubGroup
     */    
    public int getLength() {
        return length;
    }

    /**
     * Returns the start position of the SubGroup, i.e.
     * the index of the byte at which it begins in the source code.
     * 
     * @return the start offset of the SubGroup
     */
    public int getOffset() {
        return offset;
    }

    /**
     * Returns the column on which the SubGroup starts.
     * 
     * @return the column on which the SubGroup starts
     */    
    public int getStartCol() {
        return startCol;
    }

    /**
     * Returns the line on which the SubGroup starts.
     * 
     * @return the line on which the SubGroup starts
     */
    public int getStartLine() {
        return startLine;
    }    

    /**
     * Returns whether this SubGroup is empty, i.e.
     * does not contain any TokenNodes.
     * 
     * @return <code>true</code> if there are no TokenNodes
     * in the SubGroup, otherwise <code>false</code>;
     */
    public boolean isEmpty() {
    	return getTokenNodeList().isEmpty();
    }
    
    /**
     * <p>Returns the list of TokenNodes.</p>
     * 
     * <p>Note that not the actual List is returned but
     * only a copy. The TokenNodes in that copied list
     * are, however, the original TokenNode objects.</p>
     * 
     * @return the list of TokenNodes
     */
    public List<TokenNode> getTokenNodeList() {
        return new LinkedList<TokenNode>(nodeList);
    }
    
    /**
     * <p>Returns the first TokenNode of the SubGroup</p>
     * 
     * <p>If the SubGroup is empty, <code>null</code> is returned.</p>
     * 
     * @return the first TokenNode or <code>null</code> if the
     * SubGroup is empty
     */
    public TokenNode getFirstTokenNode() {
        if (isEmpty()) {
            return null;
        }
    	List<TokenNode> tnList = getTokenNodeList();
    	return tnList.get(0);
    }
    
    /**
     * <p>Returns the last TokenNode of the SubGroup</p>
     * 
     * <p>If the SubGroup is empty, <code>null</code> is returned.</p>
     * 
     * @return the last TokenNode or <code>null</code> if the
     * SubGroup is empty
     */    
    public TokenNode getLastTokenNode() {
        if (isEmpty()) {
            return null;
        }
        List<TokenNode> tnList = getTokenNodeList();
    	return tnList.get(tnList.size() - 1);
    }
    
    /**
     * Returns the number of TokenNodes in this SubGroup.
     * 
     * @return the number of TokenNodes
     */
    public int getTokenNodeCount() {
        return nodeList.size();
    }
    
    /**
     * <p>Adds a TokenNode to this SubGroup.</p>
     * 
     * <p>Adding a TokenNode is usually only done directly
     * after the creation of the SubGroup. Note that TokenNodes
     * are expected to be added in the same order 
     * as they appear in the source code. No automatic sorting
     * is performed and no checks for left-out TokenNodes are
     * done.</p>
     * 
     * <p>After a node has been added, the position information
     * of the SubGroup is updated.</p>
     * 
     * @param node the TokenNode to be added
     */
    public void addTokenNode(TokenNode node) {
        nodeList.add(node);
        updateOffsets();
    }
    
    public void addTokenNodes(List<TokenNode> nodes) {
    	nodeList.addAll(nodes);
    	updateOffsets();
    }
    
    /**
     * <p>Sets the fields for the SubGroup's position information
     * according to the position information of the contained
     * TokenNodes.</p>
     * 
     * <p>This method must not be called if the SubGroup
     * is empty</p> 
     */
    private void updateOffsets() {
        //no check for null after getFirst- and getLastTokenNode(),
        //because this method is only allowed to be executed 
        //for non-empty SubGroups
        Token firstToken = getFirstTokenNode().getToken();
        Token lastToken = getLastTokenNode().getToken();
        if (getTokenNodeCount() > 0) {
            startLine = firstToken.getLine();
            startCol = firstToken.getCol();
            endLine = lastToken.getEndLine();
            endCol = lastToken.getEndCol();
            offset = firstToken.getOffset();
            length = lastToken.getOffset() + lastToken.getLength() - firstToken.getOffset();
        } 
    }    
    
 
    /**
     * <p>Converts the TokenNodes of the SubGroup to XML and appends
     * them to the <code>parent</code> element. If there is whitespace
     * between 2 TokenNodes, it is added to the <code>parent</code> element,
     * too.</p>
     * 
     * <p>This is the last step in the process of the XML output creation.
     * To get correct results, <code>parent</code> should be an XML
     * element that represents this SubGroup.</p>
     * 
     * @param parent the XML element to which the XML representations
     * of the TokenNodes are added
     */
    public void makeXmlFromTokenNodes(Element parent) {
        
        if (!nodeList.isEmpty()) {
            Document doc = parent.getOwnerDocument();
            int i = 0;
            
            TokenNode nextNode = nodeList.get(i);
            Token nextToken = nextNode.getToken();
            
            parent.appendChild(nextNode.makeXml(doc));
            i++;
            //a loop with a counter is used here instead of a for-loop, 
            //because 2 tokens are used in each iteration and not only 1;
            //accessing the next token in a for-loop is more difficult
            while (i < nodeList.size()) {
                Token lastToken = nextToken;
                nextNode = nodeList.get(i);
                nextToken = nextNode.getToken();
                String spaces = nextNode.computeSpacesBetweenPositions(
                      lastToken.getEndLine(), lastToken.getEndCol(),
                      nextToken.getLine(), nextToken.getCol());
                parent.appendChild(doc.createTextNode(spaces));
                parent.appendChild(nextNode.makeXml(doc));
                i++;
            }
        }
    }
  

    /**
     * Returns a String representation of this SubGroup
     * 
     * @return String the String representation
     */
    public String toString() {
        return nodeList.toString();
    }
  
    public boolean _equals(Object o) {
        if (!(o instanceof SubGroup)) {
            return false;
        }
        SubGroup sg2 = (SubGroup)o;
        return nodeList.equals(sg2.nodeList) 
                && startLine == sg2.startLine
                && startCol == sg2.startCol
                && endLine == sg2.endLine
                && endCol == sg2.endCol
                && offset == sg2.offset
                && length == sg2.length
                && isStart == sg2.isStart
                && isEnd == sg2.isEnd;
    }

}
