package org.deft.repository.ast.decoration;

import java.util.LinkedList;
import java.util.List;

import org.deft.repository.ast.TokenNode;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Text;

public class SubGroupNode {
	
	private Group group;
	private SubGroup subGroup;
	private List<SubGroupNode> children = new LinkedList<SubGroupNode>();
	private int firstTokenIndex = -1;
	private int lastTokenIndex = -1;
	private SubGroupNode parent;
	
	public SubGroupNode(SubGroup subGroup, Group group, int firstTokenIndex, int lastTokenIndex) {
		this.subGroup = subGroup;
		this.group = group;
		this.firstTokenIndex = firstTokenIndex;
		this.lastTokenIndex = lastTokenIndex;
	}
	
	public Group getGroup() {
		return group;
	}
	
	public SubGroup getSubGroup() {
		return subGroup;
	}
	
	public int getFirstTokenIndex() {
		return firstTokenIndex;
	}
	
	public int getLastTokenIndex() {
		return lastTokenIndex;
	}
	
	public void addChild(SubGroupNode child) {
		children.add(child);
		child.parent = this;
	}
	
	public SubGroupNode getChild(int index) {
		return children.get(index);
	}
	
	public int getChildCount() {
		return children.size();
	}
	
	public SubGroupNode getPrecedingSibling() {
		if (hasPrecedingSibling()) {
			return parent.getChild(getChildIndex() - 1);
		} else {
			return null;
		}
	}
	
	
	public boolean hasPrecedingSibling() {
		if (parent == null) {
			return false; //root, no sibling
		}
		if (getChildIndex() == 0) {
			return false; //leftmost child, no preceding sibling
		}
		return true;
	}
	
	public SubGroupNode getFollowingSibling() {
		if (hasFollowingSibling()) {
			return parent.getChild(parent.getChildCount() + 1);
		} else {
			return null;
		}
	}


	public boolean hasFollowingSibling() {
		if (parent == null) {
			return false; //root, no sibling
		}
		if (getChildIndex() == parent.getChildCount() - 1) {
			return false; //rightmost child, no preceding sibling
		}
		return true;
	}

	
	public int getChildIndex() {
		if (parent == null) {
			return -1;
		} else {
			return parent.children.indexOf(this);
		}
	}

	
	public boolean hasChildren() {
		return getChildCount() > 0;
	}
	
	public List<SubGroupNode> getLeafNodes() {
		List<SubGroupNode> leafNodes = new LinkedList<SubGroupNode>();
		if (!hasChildren()) {
			leafNodes.add(this);
		} else {
			for (SubGroupNode child : children) {
				leafNodes.addAll(child.getLeafNodes());
			}
		}
		return leafNodes;
	}
	
	


	
	public String computeSpacesBeforeSubGroup() {
		String spaces = "";
		if (hasPrecedingSibling()) {
			spaces = computeSpacesBetweenPrecedingSiblingAndThis();
		} else {
			spaces = computeSpacesBetweenFileStartAndThis();
		}
		return spaces;
	}
	
	private String computeSpacesBetweenPrecedingSiblingAndThis() {
		SubGroupNode precedingSibling = getPrecedingSibling();
		SubGroup precedingSubGroup = precedingSibling.getSubGroup();
		TokenNode t1 = precedingSubGroup.getLastTokenNode();
		TokenNode t2 = subGroup.getFirstTokenNode();
		String spaces = t1.computeSpacesBetweenPositions(t1.getEndLine(), t1.getEndCol(), t2.getStartLine(), t2.getStartCol());
		return spaces;
	}
	
	private String computeSpacesBetweenFileStartAndThis() {
		TokenNode t = subGroup.getFirstTokenNode();
		String spaces = t.computeSpacesBetweenPositions(1, 1, 1, t.getStartCol());
		return spaces;
	}
	

	public Element makeXml(Element parent) {
        Document doc = parent.getOwnerDocument();
        
        Element e = null;
        
        if (group.isDummy()) {
        	e = doc.createElement("_dummy_");
        } else {
        	e = group.makeXmlTag(doc);
            if (subGroup.isStart()) {
                e.setAttribute("groupstart", "true");
            }
            if (subGroup.isEnd()) {
                e.setAttribute("groupend", "true");
            }
        }
        if (spacesBeforeSubGroupNecessary()) {
        	String spaces = computeSpacesBeforeSubGroup();
        	Text textNode = doc.createTextNode(spaces);
        	parent.appendChild(textNode);
        }
        
        parent.appendChild(e);
        if (hasChildren()) {
        	for (SubGroupNode child : children) {
        		child.makeXml(e);
        	} 
        } else {
        	subGroup.makeXmlFromTokenNodes(e);		
        }
        return e;
	}

	private boolean spacesBeforeSubGroupNecessary() {
		if (isRootAndHasNoCildren()) {
			return true;
		}
		if (isRoot()) {
			return false;
		}
		return parentIsRoot() || !doOtherSubGroupsHaveTheSameFirstTokenIndex();
	}
	
	private boolean isRootAndHasNoCildren() { //no decorators have been applied
		boolean isRoot = parent == null;
		return isRoot && !hasChildren();
	}

	private boolean isRoot() { 
		boolean isRoot = parent == null;
		return isRoot;
	}
	
	private boolean parentIsRoot() {
		boolean parentIsRoot = !isRoot() && parent.isRoot();
		return parentIsRoot;
	}
	
	private boolean doOtherSubGroupsHaveTheSameFirstTokenIndex() {
		int parentFirstTokenIndex = parent.getFirstTokenIndex();
		return parentFirstTokenIndex == firstTokenIndex;
	}
	
	public String toString() {
		return subGroup.toString();
	}
	
}
