//blubb
using System;
using System.Collections.Generic;
using System.Text;
using Net.Langsam.Shared;
using System.Net;
using System.Runtime.Remoting;
using System.Configuration;
using System.Runtime.Remoting.Channels.Tcp;
using System.Runtime.Remoting.Channels;
using System.Threading;
using System.IO;
using Net.Langsam.TestInitiator;
using System.ServiceProcess;
using System.Net.Sockets;
using System.Reflection;
using log4net;
using log4net.Config;
using System.Runtime.CompilerServices;
using System.Xml;
using System.Diagnostics;


namespace Net.Langsam.Agent {

    /// <summary>
    ///  An enumeration which contains all possible states of the Agent.
    /// </summary>
    enum AgentStat {
        STARTING,
        SLEEPING,
        FETCHING,
        READY,
        TESTING,
        CLEANING
    }

    /// <summary>
    /// A delegate which should be called, whenever a fatal error occured and the Agent
    /// should exit.
    /// </summary>
    delegate void ErrorHandler();

    /// <summary>
    /// This is the Agent. The Agent is available to remote hosts - this should probably
    /// be the Scheduler.
    /// </summary>
    public class Agent : MarshalByRefObject, IAgentRemote {

        /// <summary>
        /// Register error handler to this event. The method which is
        /// registered gets called if an fatal error occured.
        ///
        /// TODO:
        /// If there is a problem while fetching data or while testing,
        /// the Agent should communicate this to the Scheduler. So we
        /// need a method like scheduler.AgentCannotContinue() or s.th
        /// like that.
        /// </summary>
        internal static event ErrorHandler onError;

        private ISchedulerRemote scheduler;

        private TestObject testObject;

        private AgentState state;

        private AgentData agentData;

        private String schedulerHostname;

        private int schedulerPortnum;

        private int agentPortnum;

        private String assemblySavePath;

        private AppDomain appDomain;

        /// <summary>
        /// The Thread which fetches all Assemblies.
        /// </summary>
        private Thread fetchThread;


        /// <summary>
        /// The log4net object which should be used in the whole Agent
        /// </summary>
        internal static ILog log;

        /// <summary>
        /// Initializes the Agent with the ip address of the computer it is running on.
        /// </summary>
        public Agent() {
            state = AgentState.STARTING;

            // Check if App.config is valid xml
            XmlDocument xDoc = new XmlDocument();
            try{
                xDoc.Load(AppDomain.CurrentDomain.BaseDirectory + "Net.Langsam.Agent.exe.config");
            }
            catch (XmlException){
                if (!EventLog.SourceExists("Net.Langsam.Agent")) {
                    EventLog.CreateEventSource("Net.Langsam.Agent", "Application");
                    Console.WriteLine("CreatingEventSource");
                }

                // Create an EventLog instance and assign its source.
                EventLog errorLog = new EventLog();
                errorLog.Source = "Net.Langsam.Agent";

                // Write an informational entry to the event log.
                errorLog.WriteEntry("The configuration file is not well formed XML. Please check config file and restart the agent!", EventLogEntryType.Error);
                Console.WriteLine("The configuration file is not well formed XML. Please check config file and restart the agent!");
                onError();
                return;
            }

            // Load log4net configuration
            log = LogManager.GetLogger("Agent");
            XmlConfigurator.Configure(
                new FileInfo(AppDomain.CurrentDomain.BaseDirectory + "Net.Langsam.Agent.exe.config"));

            log.Info("The Agent is starting.");
            // Read settings from App.config. If something is wrong, exit Application.
            if (!checkSettings()) {
                onError();
                return;
            }

            // Setting agent data
            agentData = new AgentData();
            agentData.ServerPort = agentPortnum;
            agentData.Hostname = Dns.GetHostName();

            if (!openServerChannel()) {
                log.Fatal("Could not open server channel (Port: "+agentPortnum+").");
                onError();
                return;
            }
            log.Debug("Opened a port ("+agentPortnum+") for incoming connections.");

            if (!connectToScheduler()) {
                log.Info("Could not register to Scheduler ("+schedulerHostname+"). Waiting for Scheduler to connect to me.");
            }
            else {
                log.Debug("Registered myself to the Scheduler ("+schedulerHostname+").");
            }

            state = AgentState.SLEEPING;
            log.Debug("Now in state: " + state + ".");
        }


        /// <summary>
        /// Tries to connect and register to the Scheduler. This will be tried 3 times with a timeout
        /// of 10 seconds. If this fails false will be returned.
        /// </summary>
        /// <returns>
        /// False if the connection failed.
        /// </returns>
        private void connectToScheduler() {

            String uri = "tcp://"
                + schedulerHostname
                + ":"
                + schedulerPortnum
                + "/Scheduler";

            try {
                scheduler = (ISchedulerRemote)Activator.GetObject(typeof(ISchedulerRemote), uri);
                if (scheduler.RegisterAgent(agentData)) {
                    return true;
                }
                else {
                    log.Debug("Cannot register to Scheduler. Scheduler signalized a problem.");
                    return false;
                }
            }
            catch (Exception) {
                log.Debug("Cannot register to Scheduler. Cannot connect to Scheduler.");
                return false;
            }
        }


        /// <summary>
        /// Reads setting from App.config and initializes class variables based upon this data.
        /// If there are invalid enties which prevents the Agent from running correctly - Exit the Application.
        /// </summary>
        /// <returns>
        /// False if some setting could not be read or if they are invalid.
        /// </returns>
        private bool checkSettings() {

            IPAddress[] schedulerIpAddresses;

            AppSettingsReader settings = new AppSettingsReader();


            /* Check schedulerHostname */
            try {
                schedulerHostname = (String)settings.GetValue("SchedulerHostName", typeof(String));
            }
            catch (Exception e) {
                log.Fatal("Hostname of Scheduler not set in config file.", e);
                return false;
            }
            if (schedulerHostname == null || schedulerHostname == "") {
                log.Fatal("Hostname of Scheduler set to empty string.");
                return false;
            }


            /* Check schedulerHostname */
            try {
                schedulerPortnum = (int)settings.GetValue("SchedulerPortnum", typeof(int));
            }
            catch (Exception e) {
                log.Fatal("Portnumber of Scheduler not set in config file.", e);
                return false;
            }


            /* Check agentPortnum */
            try {
                agentPortnum = (int)settings.GetValue("AgentPortnum", typeof(int));
            }
            catch (Exception e) {
                log.Fatal("Portnumber of Agent not set in config file.", e);
                // Exit Application
                return false;
            }


            /* Check assemblySavePath */
            try {
                assemblySavePath = (String)settings.GetValue("AssemblySavePath", typeof(String));
            }
            catch (Exception e) {
                log.Fatal("assemblySavePath of Agent not set in config file.", e);
                return false;
            }
            if (assemblySavePath == "") {
                log.Fatal("AssemblySavePath of Agent not set to empty string config file.");
                return false;
            }


            /* Create assemblySavePath */
            if (!Directory.Exists(assemblySavePath)) {
                try {
                    Directory.CreateDirectory(assemblySavePath);
                }
                catch (Exception e){
                    log.Fatal("Could not create directory " + assemblySavePath + ".", e);
                    return false;
                }
            }


            /* Resolve Scheduler hostname */
            schedulerIpAddresses = Dns.GetHostAddresses(schedulerHostname);
            if (schedulerIpAddresses == null) {
                log.Fatal("Could not resolve Scheduler hostname.");
                return false;
            }
            else if (schedulerIpAddresses[0] == null) {
                log.Fatal("Could not resolve Scheduler hostname.");
                return false;
            }

            // If there were no errors
            return true;
        }


        /// <summary>
        /// Open own server channel. To this channel the Scheduler may connect.
        /// Must be called before connectToScheduler.
        /// </summary>
        /// <returns>
        /// False, if no server channel could be opened.
        /// </returns>
        private bool openServerChannel() {

            try {
                log.Debug("Opening local port.");
                TcpChannel serverChannel = new TcpChannel(agentData.ServerPort);
                ChannelServices.RegisterChannel(serverChannel, false);
                // Distribute agent object as "Agent"
                RemotingServices.Marshal(this, "Agent");

            } catch (Exception e) {
                log.Fatal("Could not open local server port ("+agentPortnum+").", e);
                return false;
            }

            log.Debug("Local port opened successfully.");
            return true;
        }


        /// <summary>
        /// Fetches and unzips data from the Scheduler. This method should run
        /// in its own thread cause it shurely will take a while.
        /// </summary>
        private void fetchData() {

            log.Debug("Now in state: " + state + ".");

            // Get location from Scheduler
            log.Debug("Getting AssemblyLocation from Scheduler.");
            AssemblyLocation location = scheduler.GetAssemblyLocation();

            IAssemblyFetcher fetcher = null;

            // scheduler could retrun null the connectionType entry in the app config is wrong
            if (location != null) {

                if (location.ConnType == AssemblyLocation.ConnectionType.LOCAL) {
                    log.Debug("Using LocalFetcher.");
                    fetcher = new AssemblyLocalFetcher();
                } else if (location.ConnType == AssemblyLocation.ConnectionType.SOCKET) {
                    log.Debug("Using NetworkFetcher.");
                    // Set location to Scheduler address
                    location.Hostname = schedulerHostname;
                    fetcher = new AssemblyNetworkFetcher();
                }

            } else {
                log.Fatal("No matching IAssemblyFetcher found.");
                Abort();
                return;
            }



            // Fetch file
            if (fetcher.fetch(location, assemblySavePath)) {
                log.Debug("Assemblies fetched successfully.");
                // Signalize the Scheduler, that we are ready to start the Test.
                if (scheduler.AgentReady(agentData)) {
                    log.Debug("Told the Scheduler that I'm ready.");
                } else {
                    log.Error("Could not tell Scheduler that I'm ready.");
                    Abort();
                    return;
                }
                state = AgentState.READY;
                log.Debug("Now in state: " + state + ".");
            }
            else {
                log.Error("Fetching of assemblies failed.");
                Abort();
                return;
            }

            testObject = scheduler.GetTestObject();
            if (testObject == null) {
                log.Error("Didn't receive a TestObject.");
                Abort();
                return;
            }
            return;
        }


        /// <summary>
        /// Create a new AppDomain, in which the real testing suff will run.
        /// </summary>
        private void createNewAppDomain() {

            String basePath = AppDomain.CurrentDomain.BaseDirectory;

            try {
                AppDomainSetup setup = new AppDomainSetup();
                setup.ApplicationBase = basePath;
                setup.PrivateBinPath = basePath;
                setup.ApplicationName = "TestInitiator";
                setup.ShadowCopyFiles = "true";

                appDomain = AppDomain.CreateDomain("TestInitiator Domain", null, setup);

                Tester tester = (Tester)appDomain.CreateInstanceFromAndUnwrap(
                    basePath + "Net.Langsam.TestInitiator.dll",
                    "Net.Langsam.TestInitiator.Tester");

                // Set data on the new AppDomain
                tester.AgentData = agentData;
                tester.AssemblySavePath = assemblySavePath;
                tester.Scheduler = scheduler;
                tester.TestObject = testObject;

                // Start test execution in new AppDomain
                tester.runTest();
            }

            catch (AppDomainUnloadedException e) {
                log.Fatal("Failed to access AppDomain, because it was unloaded. Maybe the test was aborted due to a timer.", e);
                // Tell Scheduler that we're gone.
                scheduler.AgentFailedTesting(agentData.Hostname);
            }

            catch (Exception e) {
                log.Fatal("Could not create AppDomain. Maybe too many threads running or not enough memory left.", e);
                // Tell Scheduler that we're gone.
                scheduler.AgentFailedTesting(agentData.Hostname);
            }

            // Cleanup temp files
            finally {
                if (appDomain != null) {
                    log.Debug("Unloading AppDomain.");
                    AppDomain.Unload(appDomain);
                    appDomain = null;
                }

                state = AgentState.CLEANING;
                log.Debug("Now in state " + state + ".");

                delTempFiles();
                state = AgentState.SLEEPING;
                log.Debug("Now in state " + state + ".");
            }
        }


        /// <summary>
        /// Deletes previously downloaded files.
        /// </summary>
        /// <returns>
        /// False if some files could not be deleted.
        /// </returns>
        private bool delTempFiles() {

            if (Directory.Exists(assemblySavePath)) {

                try {
                    log.Debug("Cleaning up temp. files.");
                    Directory.Delete(assemblySavePath, true);
                    log.Debug("Successfully cleaned up temp files.");
                } catch (Exception e) {
                    log.Error("Could not delete all temp. files.", e);
                    return false;
                }
            }
            return true;
        }


        /// <summary>
        /// Logs of from the Scheduler. This method should only be called by
        /// AgentService.OnStop().
        /// </summary>
        /// <returns>
        /// False, if there was an error, deregistering.
        /// </returns>
        internal bool DeRegister() {
            return scheduler.DeRegisterAgent(agentData);
        }


        #region IAgentRemote Members


        /// <summary>
        /// This method gets called by the Scheduler and tells the Agent to fetch
        /// the Test data.
        /// Should not do something, if the Agent has'nt registered itselfe yet.
        /// </summary>
        /// <returns></returns>
        [MethodImpl(MethodImplOptions.Synchronized)]
        public bool Wakeup() {

            log.Info("I got a wakeup() from the Scheduler. Start fetching data.");
            lock (this) {
                if (state != AgentState.SLEEPING) {
                    log.Error("Not in state SLEEPING as expected. In state: " + state);
                    return false;
                }
                else {
                    state = AgentState.FETCHING;
                }
            }


            // Create a thread to fetch data from the Scheduler. If we would not use
            // an extra thread, the Wakeup() method would block until all data is
            // fetched. That would be bad!
            ThreadStart threadStart = new ThreadStart(fetchData);
            fetchThread = new Thread(threadStart);
            fetchThread.Name = "fetchThread";

            try {
                fetchThread.Start();
            }
            catch (Exception e) {
                log.Error("Could not start fetchThread.", e);
                Abort();
                return false;
            }

            return true;
        }


        /// <summary>
        /// This methods gets called by the Scheduler and tells the Agent to start
        /// the actual Test.
        /// Should not do something, if the Agent hasn't fetched all data yet.
        /// </summary>
        /// <returns>
        /// False, if the Test could not be started.
        /// </returns>
        [MethodImpl(MethodImplOptions.Synchronized)]
        public bool StartTest() {

            log.Info("I got a StartTest() from the Scheduler. Trying to start the test.");

            // First check, if all files where fetched correctly
            log.Debug("Waiting for fetchThraed to return.");
            fetchThread.Join();
            log.Debug("FetchThread has returned.");

            if (state != AgentState.READY) {
                log.Error("Not in state READY. In state " + state + ".");
                Abort();
                return false;
            }
            else {
                state = AgentState.TESTING;
            }

            // We have to do the test in a seperate AppDomain. An AppDomain
            // is a kind of seperate process. This is sadly needed as .NET
            // does not support Assembly.Unload(). So the only possible solution
            // is to load the Assemblies in a seperate AppDomain and after
            // we've finished kill the whole AppDomain. After that the referenced
            // Assemblies are no longer locked.
            Thread t = new Thread(new ThreadStart(createNewAppDomain));
            try {
                t.Start();
            } catch (Exception e) {
                log.Fatal("Could not start Test.", e);
                Abort();
                return false;
            }
            return true;
        }


        /// <summary>
        /// If this method is called, the Agent should stop working, clean up and go to
        /// sleep again. So after calling Abort() the Agent should be in its initial state.
        /// Deletes all previously fetched binaries.
        /// While cleanup the state gets changed to CLEANING, after
        /// finishing the state is set to SLEEPING.
        /// </summary>
        /// <returns>
        /// False, if there where errors stopping threads.
        /// </returns>
        [MethodImpl(MethodImplOptions.Synchronized)]
        public bool Abort() {

            if (log != null) {
                log.Debug("Abort() was called.");
            }

            if (state == AgentState.SLEEPING) {
                if (log != null) {
                    log.Warn("Already in state SLEEPING. Not cleaning up.");
                }
                return true;
            }

            state = AgentState.CLEANING;
            if (log != null) {
                log.Debug("Now in state " + state + ".");
            }

            bool result = true;

            try {
                if (fetchThread.IsAlive) {
                    if (log != null) {
                        log.Debug("Killing fetchThread.");
                    }
                    fetchThread.Abort();
                    if (log != null) {
                        log.Debug("Killed fetchThread successfully.");
                    }
                }

            } catch (Exception e) {
                if (log != null) {
                    log.Warn("Could not kill fetchThread.", e);
                }
                result = false;
            }

            if (appDomain != null) {
                try {
                    if (log != null) {
                        log.Debug("Killing TestInitiator AppDomain.");
                    }
                    AppDomain.Unload(appDomain);
                    appDomain = null;
                    if (log != null) {
                        log.Debug("Killed TestInitiator AppDomain.");
                    }
                }
                catch (Exception e) {
                    if (log != null) {
                        log.Warn("Could not kill TestInitiator AppDomain.", e);
                    }
                    result = false;
                }
            }

            if (!delTempFiles()) {
                result = false;
            }

            state = AgentState.SLEEPING;
            if (log != null) {
                log.Debug("Now in state " + state + ".");
            }

            return result;
        }


        /// <summary>
        /// Returns AgentData. This method is also used for verifying that the Agent
        /// is online.
        /// </summary>
        /// <returns></returns>
        [MethodImpl(MethodImplOptions.Synchronized)]
        public AgentData GetAgentData() {
            log.Debug("GetAgentData() was called.");
            return agentData;
        }


        #endregion


        /// <summary>
        /// Returning null here to set the lifetime of the Agent object to infinite.
        /// </summary>
        /// <returns>
        /// Allways null.
        /// </returns>
        public override object InitializeLifetimeService() {
            return null;
        }
    }
}