package org.deft.repository.ast;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;

import org.apache.commons.jxpath.JXPathContext;
import org.apache.log4j.Logger;
import org.deft.repository.Util;
import org.deft.repository.ast.decoration.Ident;
import org.deft.repository.ast.decoration.NodeInformation;
import org.deft.repository.ast.jxpath.TreeNodePointer;
import org.deft.repository.ast.jxpath.TreeNodeRoot;
import org.deft.repository.ast.transform.ITreeNodeFilter;



/**
 * A TreeNode is the internal representation of a node in the Abstract Syntax Tree (AST) of a source file.
 * 
 * After the source files have been parsed an XML-AST is produced. XML is a good format for applying the
 * query. However, after the query has been executed, several postprocessing steps are necessary for
 * formatting purposes. This includes adding several types of information to the nodes. XML is not suitable
 * for that task, therefore the XML-AST is converted to a tree of TreeNodes, which can be given {@link NodeInformation}.
 * 
 * @author Andreas Bartho
 */
public class TreeNode implements Iterable<TreeNode> {

    protected String name;
    protected List<TreeNode> children = new LinkedList<TreeNode>();
    protected TreeNode parent;
    protected TreeNodeRoot tnr;
    
    private int startLine, startCol, endLine, endCol, offset, endOffset, length;
    private boolean lock;
    protected List<NodeInformation> informationList = new LinkedList<NodeInformation>();
    private static TokenNodeFilter tokenNodeFilter;
    private static TokenNodeComparator tokenNodeComparator = new TokenNodeComparator();
    private static Logger logger = Logger.getLogger(TreeNode.class);
    
    private TreeNodePointer pointer = null;
    
    /**
     * You should not use this constructor. It is for serializing purposes only.
     */
    public TreeNode(){
    	name = "TreeNode";
    }
    
    
    
    /**
     * Creates from an XML element the belonging TreeNode.
     * 
     * @param e
     */
//    public TreeNode(Node node) {
//    	name = node.getNodeName();
//        if (node == null) {
//            throw new IllegalArgumentException("Cannot create a Node from a null Element.");
//        }
//        if (name == null || name.equals("")) {
//            throw new IllegalArgumentException("TreeNode name must neither be null nor the empty string.");
//        }        
//        this.name = node.getNodeName();
//    }
    
    public void setTnr(TreeNodeRoot tnr) {
		this.tnr = tnr;
	}



	/**
     * Creates a TreeNode with a given name.
     * @param name
     */
    public TreeNode(String name) {
        if (name == null || name.equals("")) {
            throw new IllegalArgumentException("TreeNode name must neither be null nor the empty string.");
        }
        this.name = name;
        tokenNodeFilter = new TokenNodeFilter();
    }
    public void addChild(TreeNode n) {
        addChild(children.size(), n);
    }
    
    public void addChild(int position, TreeNode n) {
        if (n.parent != null) {//node is already used in a tree, remove it from old position
            n.parent.children.remove(n);
        }
        n.parent = this;
        children.add(position, n);
    }
    
    public void removeChild(TreeNode n){
    	if (children.contains(n)){
    		children.remove(n);
    		n.parent = null;
    	}
    }
         
    public boolean replaceChild(TreeNode oldNode, TreeNode newNode){
    	if (children.contains(oldNode) && !children.contains(newNode)){   		
    		int index = children.indexOf(oldNode);
    		children.remove(index);
    		newNode.parent = this;
    		children.add(index,newNode); 
    		return true;
    		
    	}
    	return false;
    }
    
    public String getName() {
        return name;
    }
    
    public TreeNode getParent() {
        return parent;
    }
    
    public TreeNodePointer getPointer() {
    	if (pointer == null) {
    		TreeNode node = getParent();
    		if (node == null) {
    			node = getTreeNodeRoot();
    		}
    		assert node != null : "Could not get Pointer for TreeNode, TreeNodeRoot does not exist";
    		pointer = new TreeNodePointer(node.getPointer(), this);
    	}
    	return pointer;
    }

    public TreeNodeRoot getTreeNodeRoot() {
    	TreeNode node = this;
    	while ((node.getParent() != null) && (node.tnr == null))
    		node = node.getParent();
    	assert node.tnr != null : "Tree has no TreeNodeRoot";
    	return node.tnr;
    }
    
    /**
     * Returns whether this node is the root node. A node is the root node
     * of a tree if it has a TreeNodeRoot as its parent. A TreeNodeRoot is 
     * a virtual node which has only been added for XPath evaluation, therefore
     * its child (i.e. this node) counts as actual root.
     * 
     * @return
     */
    public boolean isRoot() {
    	TreeNode parent = getParent();
    	return parent != null && parent instanceof TreeNodeRoot; 
    }
    
    public List<TreeNode> getChildren() {
        return children;
    }
       
    public TreeNode getChild(int position) {
        return children.get(position);
    }
    
    public List<TreeNode> getChildByName(String name) {
    	List<TreeNode> retList = new LinkedList<TreeNode>();
    	for (TreeNode child : children) {
    		if (child.getName().equals(name)) {
    			retList.add(child);
    		}
    	}
    	return retList;
    }
    
    public TreeNode getFirstChild() {
        if (getChildCount() == 0) {
            return null;
        }        
        return children.get(0);
    }
    
    public TreeNode getLastChild() {
        if (getChildCount() == 0) {
            return null;
        }
        return children.get(getChildCount() - 1);
    }

    
    private List<TreeNode> getSiblingsAndSelf() {
    	if (parent == null) {
    		List<TreeNode> list = new LinkedList<TreeNode>();
    		list.add(this);
    		return list;
    	} else {
    		return parent.getChildren();
    	}
    }
    
    public TreeNode getNextSibling() {
    	List<TreeNode> siblings = getSiblingsAndSelf();
    	int pos = getSiblingPosition();
    	if (pos < siblings.size() - 1) {
    		return siblings.get(pos + 1);
    	} else {
    		return null;
    	}
    }
    
    public TreeNode getPreviousSibling() {
    	List<TreeNode> siblings = getSiblingsAndSelf();
    	int pos = getSiblingPosition();
    	if (pos > 0) {
    		return siblings.get(pos - 1);
    	} else {
    		return null;
    	}
    }
    
    public int getChildCount() {
        return children.size();
    }
    
    public int getSiblingPosition() {
        return getSiblingsAndSelf().indexOf(this);
    }
    
    public boolean hasChildren() {
        return !getChildren().isEmpty(); 
    }
    

    
    public void serialize(List<TokenNode> list) {
    	TreeWalker tw = new TreeWalker(this, tokenNodeFilter);
    	while (tw.hasNext()) {
    		TreeNode tn = tw.next();
        	list.add((TokenNode)tn);
    	}
        Collections.sort(list, tokenNodeComparator);
    }
    
    public List<TokenNode> serialize() {
    	List<TokenNode> list = new LinkedList<TokenNode>();
    	serialize(list);
    	return list;
    }
    
    public String toString() {
        return "TreeNode: " + name;
    }
    
    public Position getPosition() {
    	return new Position(getStartLine(),
    			getStartCol(),
    			getEndLine(),
    			getEndCol(),
    			getOffset(),
    			getLength());
    }
    
    public int getStartLine() {
        return lock ? startLine : getFirstChild().getStartLine();
    }
    
    public int getStartCol() {
        return lock ? startCol : getFirstChild().getStartCol();
    }
    
    public int getEndLine() {        
        return lock ? endLine : getLastChild().getEndLine();
    }
    
    public int getEndCol() {
        return lock ? endCol : getLastChild().getEndCol();
    }
    
    public int getOffset() {
        return lock ? offset : getFirstChild().getOffset();
    }
    
    public int getEndOffset() {
        return lock ? endOffset : getLastChild().getEndOffset();
    }    
    
    public int getLength() {
        return lock ? length : getEndOffset() - getOffset();
    }
  
    
    public void lock() {
        this.startLine = getStartLine();
        this.endLine = getEndLine();
        this.startCol = getStartCol();
        this.endCol = getEndCol();
        this.offset = getOffset();
        this.endOffset = getEndOffset();
        this.length = getLength();
        lock = true;
        for (TreeNode child : children) {
            child.lock();
        }
    }
    
    public void unlock() {
        lock = false;
        for (TreeNode child : children) {
            child.unlock();
        }        
    }
    
    public boolean isLocked() {
        return lock;
    }
    
    public List<NodeInformation> getInformation() {
        return informationList;
    }    
    
    public NodeInformation getInformation(Ident ident) {
        //return getInformation(ident.getIdString());
        for (NodeInformation information : informationList) {
            if (information.getIdent().equals(ident)) {
                return information;
            }
        }
        return null;
    }
    
    public NodeInformation getInformation(String ident) {
        for (NodeInformation information : informationList) {
            if (information.getIdent().getIdString().equals(ident)) {
                return information;
            }
        }
        return null;
    }
    
    public boolean hasInformation(Ident ident) {
    	return getInformation(ident) != null;
    }
    
    public boolean hasInformation(String ident) {
    	return getInformation(ident) != null;
    }
    
    public void addInformation(NodeInformation information) {
        //if there is already an information of that type, overwrite it
        NodeInformation oldInf = getInformation(information.getIdent());
        if (oldInf != null) {
            informationList.remove(oldInf);
        }
        informationList.add(information);
    }
    
    
    /**
     * Returns all descendant nodes that have a NodeInformation of type
     * <code>ident</code>. If a node is found, its descendants are not
     * examined anymore as they are supposed to carry that NodeInformation
     * implicitly.
     *  
     * @param ident
     * @return
     */
    public List<TreeNode> getDescendants(Ident ident) {
    	List<TreeNode> retList = new ArrayList<TreeNode>();
    	getDescendants(this, retList, ident);
    	return retList;
    }
    
	private void getDescendants(TreeNode rootNode, List<TreeNode> result, 
			Ident ident) {

		boolean found = false;
		for (NodeInformation info : rootNode.getInformation()) {
			if (info.getIdent().equals(ident)) {
				found = true;
			}
		}
		if (found) {
			result.add(rootNode);
		} else {
			for (TreeNode child : rootNode.getChildren()) {
				getDescendants(child, result, ident);
			}
		}
	}

	public Iterator<TreeNode> iterator() {
		return new TreeWalker(this, null);
	} 
	
	public List<TreeNode> executeXPathQuery(String query) {		
		LinkedList<TreeNode> result = new LinkedList<TreeNode>();
    	
		JXPathContext context = getTreeNodeRoot().getContext().getRelativeContext(getPointer());
		
		for (Iterator<?> iter = context.iterate(query); iter.hasNext();) {
			Object value = iter.next();
			assert(value instanceof TreeNode) : "The selected value was not an instance of TreeNode.";
			result.add((TreeNode)value);
		}
		
    	return result;
	}
	
	public String getTextContent() {
		StringBuilder sb = new StringBuilder();
		List<TokenNode> lTokens = serialize();
		TokenNode tnPrev = null;
		for (TokenNode tn : lTokens) {
			if (tnPrev != null) {
				sb.append(tnPrev.getToken().getText());
				String spaces = Util.computeSpacesBetweenPositions(
						tnPrev.getEndLine(), tnPrev.getEndCol(),
						tn.getStartLine(), tn.getStartCol());
				sb.append(spaces);
			}
			tnPrev = tn;
		}
		//add last token
		sb.append(lTokens.get(lTokens.size() - 1).getToken().getText());
		return sb.toString();
	}
	

    private class TokenNodeFilter implements ITreeNodeFilter {
		
		public boolean accept(TreeNode node) {
			return node instanceof TokenNode;
		}
    }
	

}
